#pragma once

#include "worker.h"
#include "types.h"

#include <ymod_httpclient/cluster_client.h>

#include <yplatform/reactor.h>
#include <yplatform/ptree.h>
#include <yplatform/coroutine.h>
#include <yplatform/time_traits.h>

#include <memory>
#include <mutex>

namespace NYmodHttpWatcher {

class TWatcherImpl : public yplatform::log::contains_logger {
public:
    explicit TWatcherImpl(yplatform::reactor& reactor);

    void Init(const yplatform::ptree& configuration);
    void Stop();

    void SetHandler(THandler handler);
    void SetErrorHandler(TErrorHandler handler);

private:
    void RunWorker(THttpClientPtr httpClient, std::string url, TPeriod period, TPeriod startDelay);

private:
    struct THandlers {
        std::mutex Lock;
        THandler Handler;
        TErrorHandler ErrorHandler;
    };

    yplatform::reactor& Reactor;
    boost::asio::io_context& Io;
    yplatform::time_traits::timer Timer;
    std::shared_ptr<THandlers> Handlers;
    std::shared_ptr<TWorker> Worker;
};

TWatcherImpl::TWatcherImpl(yplatform::reactor& reactor)
    : Reactor(reactor)
    , Io(*Reactor.io())
    , Timer(Io)
    , Handlers(std::make_shared<THandlers>())
{}

void TWatcherImpl::Init(const yplatform::ptree &configuration) {
    auto httpClient = std::make_shared<ymod_httpclient::cluster_client>(Reactor, configuration.get_child("http_client"));
    auto url = configuration.get<std::string>("url");
    auto period = configuration.get<TPeriod>("period");
    auto startDelay = configuration.get<TPeriod>("start_delay");

    RunWorker(std::move(httpClient), std::move(url), period, startDelay);
}

void TWatcherImpl::RunWorker(THttpClientPtr httpClient, std::string url, TPeriod period, TPeriod startDelay) {
    auto callback = [handlers = Handlers](const yhttp::response& response) {
        std::lock_guard guard(handlers->Lock);
        if (handlers->Handler) {
            handlers->Handler(response);
        }
    };
    auto errorCallback = [handlers = Handlers](const boost::system::error_code& ec) {
        std::lock_guard guard(handlers->Lock);
        if (handlers->ErrorHandler) {
            handlers->ErrorHandler(ec);
        }
    };

    Worker = std::make_shared<TWorker>(
        yhttp::request::GET(std::move(url)),
        period,
        Io,
        std::move(httpClient),
        std::move(callback),
        std::move(errorCallback));

    if (startDelay > TPeriod::zero()) {
        Timer.expires_after(startDelay);
        Timer.async_wait([worker = Worker](const boost::system::error_code& ec) {
            if (ec != boost::asio::error::operation_aborted) {
                yplatform::spawn(worker);
            }
        });
    } else {
        yplatform::spawn(Worker);
    }
}

void TWatcherImpl::Stop() {
    Timer.cancel();
    Worker->Stop();
}

void TWatcherImpl::SetHandler(THandler handler) {
    std::lock_guard guard(Handlers->Lock);
    Handlers->Handler = std::move(handler);
}

void TWatcherImpl::SetErrorHandler(TErrorHandler handler) {
    std::lock_guard guard(Handlers->Lock);
    Handlers->ErrorHandler = std::move(handler);
}

} // namespace NYmodHttpWatcher
