#pragma once

#include <ymod_httpclient/types.h>
#include <ymod_httpclient/settings.h>
#include <ymod_httpclient/errors.h>
#include <ymod_httpclient/response_handler.h>
#include <ymod_httpclient/request.h>
#include <ymod_httpclient/url_encode.h>
#include <ymod_httpclient/detail/bind.h>

#include <yplatform/coroutine.h>
#include <yplatform/task_context.h>
#include <yplatform/future/future.hpp>
#include <yplatform/time_traits.h>

#include <boost/asio/dispatch.hpp>

#include <functional>
#include <unordered_map>

namespace ymod_httpclient {

typedef yplatform::future::promise<void> promise_void_t;
typedef yplatform::future::future<void> future_void_t;

using headers_dict = std::unordered_multimap<string, string>;

struct response
{
    int status = 0;
    headers_dict headers;
    string body;
    string reason;
};

class simple_call
{
public:
    using task_context_ptr = ::ymod_httpclient::task_context_ptr;
    using callback_type = std::function<void(const boost::system::error_code&, response)>;
    using settings = ::ymod_httpclient::settings;

    // Don't call at application start/stop
    virtual response run(task_context_ptr ctx, request req) = 0;
    virtual response run(task_context_ptr ctx, request req, const options& options) = 0;

    virtual void async_run(task_context_ptr ctx, request req, callback_type callback) = 0;
    virtual void async_run(
        task_context_ptr ctx,
        request req,
        const options& options,
        callback_type callback) = 0;

    template <typename Callback>
    auto async_run(task_context_ptr ctx, request req, const options& options, Callback&& callback)
    {
        if constexpr (yplatform::is_awaitable_token<Callback>())
        {
            // Boost asio awaitable transforms error codes to exceptions
            // so we use special signature here to handle error codes instead of exceptions.
            // Usage example:
            //     auto [ec, response] = co_await client.async_run(...);
            using signature_type =
                void(std::tuple<boost::system::error_code, ymod_httpclient::response>);
            boost::asio::async_completion<Callback, signature_type> init(callback);
            // XXX Lambda will be copied inside std::function
            // but awaitable tokens are noncopyable until boost 1.70
            // so wrap callback with shared_ptr.
            auto completion_handler_shared =
                make_shared<decltype(init.completion_handler)>(std::move(init.completion_handler));
            callback_type converted_callback =
                [callback = completion_handler_shared](auto ec, auto response) mutable {
                    (*callback)(std::tuple(std::move(ec), std::move(response)));
                };
            async_run(std::move(ctx), std::move(req), options, std::move(converted_callback));
            return init.result.get();
        }
        else
        {
            using signature_type = void(boost::system::error_code, ymod_httpclient::response);
            boost::asio::async_completion<Callback, signature_type> init(callback);
            callback_type converted_callback = [callback = std::move(init.completion_handler)](
                                                   auto ec, auto response) mutable {
                boost::asio::dispatch(detail::bind(std::move(callback), ec, std::move(response)));
            };
            async_run(std::move(ctx), std::move(req), options, std::move(converted_callback));
            return init.result.get();
        }
    }

    template <typename Callback>
    auto async_run(task_context_ptr ctx, request req, Callback&& callback)
    {
        return async_run(
            std::move(ctx), std::move(req), options{}, std::forward<Callback>(callback));
    }

    virtual ~simple_call()
    {
    }
};

class call : public simple_call
{
public:
    virtual future_void_t get_url(
        task_context_ptr context,
        response_handler_ptr handler,
        const remote_point_info_ptr host,
        const string& req,
        const string& headers = "") = 0;

    virtual future_void_t head_url(
        task_context_ptr context,
        response_handler_ptr handler,
        const remote_point_info_ptr host,
        const string& req,
        const string& headers = "") = 0;

    virtual future_void_t post_url(
        task_context_ptr context,
        response_handler_ptr handler,
        const remote_point_info_ptr host,
        const string& req,
        const string_ptr& post,
        const string& headers = "",
        bool log_post_args = true) = 0;

    virtual future_void_t mpost_url(
        task_context_ptr context,
        response_handler_ptr handler,
        const remote_point_info_ptr host,
        const string& req,
        post_chunks&& post,
        const string& headers = "",
        bool log_post_args = true) = 0;

    virtual remote_point_info_ptr make_rm_info(const string& host) = 0;
    virtual remote_point_info_ptr make_rm_info(const string& host, bool reuse_connection) = 0;

    virtual remote_point_info_ptr make_rm_info(const string& host, const timeouts& timeouts) = 0;
    virtual remote_point_info_ptr make_rm_info(
        const string& host,
        const timeouts& timeouts,
        bool reuse_connection) = 0;

    virtual ~call()
    {
    }
};

class cluster_call
{
public:
    using task_context_ptr = ::ymod_httpclient::task_context_ptr;
    using callback_type = simple_call::callback_type;

    using options = cluster_client_options;

    virtual void async_run(task_context_ptr ctx, request req, callback_type callback) = 0;
    virtual void async_run(
        task_context_ptr ctx,
        request req,
        const options& options,
        callback_type callback) = 0;

    template <typename CompletionToken>
    auto async_run(
        task_context_ptr ctx,
        request req,
        const options& options,
        CompletionToken&& token)
    {
        using Signature = void(boost::system::error_code, ymod_httpclient::response);
        boost::asio::async_completion<CompletionToken, Signature> init(token);
        async_run(
            std::move(ctx),
            std::move(req),
            options,
            callback_type([handler = init.completion_handler](
                              boost::system::error_code ec, response response) {
                boost::asio::dispatch(detail::bind(std::move(handler), ec, std::move(response)));
            }));
        return init.result.get();
    }

    template <typename CompletionToken>
    auto async_run(task_context_ptr ctx, request req, CompletionToken&& token)
    {
        return async_run(
            std::move(ctx), std::move(req), options{}, std::forward<CompletionToken>(token));
    }

    virtual ~cluster_call() = default;
};

}

namespace yhttp {

using ymod_httpclient::response;
using ymod_httpclient::simple_call;
using ymod_httpclient::call;

}
