#pragma once

#include <ymod_httpclient/settings.h>
#include <random>

namespace ymod_httpclient::detail {

template <typename RandomGenerator>
class backoff_impl
{
public:
    backoff_impl(
        const backoff_settings& settings,
        time_traits::duration min_request_duration,
        RandomGenerator random_generator = {})
        : min_request_duration(min_request_duration)
        , random_generator(std::move(random_generator))
        , max_backoff_delay(
              time_traits::duration_cast<time_traits::milliseconds>(settings.max).count())
        , base_delay(time_traits::duration_cast<time_traits::milliseconds>(settings.base).count())
        , multiplier(settings.multiplier)
        , max_exp(calc_max_exp(max_backoff_delay, base_delay, multiplier))
    {
    }

    time_traits::duration calc_delay(task_context_ptr ctx, unsigned attempt_num)
    {
        if (attempt_num == 0) return time_traits::duration::zero();

        auto exp = attempt_num - 1;
        auto exponential_delay = exp > max_exp ?
            max_backoff_delay :
            static_cast<int64_t>(base_delay * pow(multiplier, exp));
        auto max_delay = std::min(exponential_delay, calc_delay_to_deadline(ctx->deadline()));
        return time_traits::milliseconds(random_generator(0l, max_delay));
    }

private:
    int64_t calc_delay_to_deadline(time_traits::time_point deadline)
    {
        using time_traits::duration_cast;
        using time_traits::milliseconds;

        auto min_deadline_point = time_traits::clock::now() + min_request_duration;
        if (deadline <= min_deadline_point) return 0;

        return duration_cast<milliseconds>(deadline - min_deadline_point).count();
    }

    double calc_max_exp(double max_backoff_delay_ms, double base_delay_ms, double multiplier)
    {
        if (base_delay_ms < 1) throw std::runtime_error("base delay is too small");
        if (multiplier < 1) throw std::runtime_error("multiplier is too small");

        auto max_power_result = static_cast<double>(max_backoff_delay_ms) / base_delay_ms;
        if (max_power_result <= 1) return 0;

        return logn(max_power_result, multiplier);
    }

    double logn(double number, double base)
    {
        return std::log(number) / std::log(base);
    }

    time_traits::duration min_request_duration;
    RandomGenerator random_generator;

    int64_t max_backoff_delay;
    double base_delay;
    double multiplier;
    double max_exp;
};

struct random_generator
{
    template <typename Int>
    Int operator()(Int min, Int max)
    {
        static thread_local std::mt19937 generator(time(0));
        return std::uniform_int_distribution<int64_t>(min, max)(generator);
    }
};

using backoff = backoff_impl<random_generator>;

}
