#pragma once

#include "backoff.h"
#include "eval_timeouts.h"
#include <ymod_httpclient/detail/balancing_call_op.h>
#include <ymod_httpclient/call.h>
#include <ymod_httpclient/response_status.h>
#include <yplatform/coroutine.h>
#include <yplatform/util/safe_call.h>
#include <random>

#include <yplatform/yield.h>

namespace ymod_httpclient::detail {

inline bool retriable(boost::system::error_code err)
{
    static const std::set<boost::system::error_code> retriable_codes = {
        http_error::connect_error,
        http_error::ssl_error,
        http_error::read_error,
        http_error::write_error,
        http_error::server_response_error,
        http_error::server_status_error,
        http_error::server_header_error,
        http_error::response_handler_error,
        http_error::session_closed_error,
        http_error::parse_response_error,
        http_error::unknown_error,
        http_error::connection_timeout,
        http_error::request_timeout,
        http_error::eof_error,
        http_error::task_throttled
    };
    return retriable_codes.count(err);
}

template <typename BalancingCallOp, typename Handler, typename Interpreter, typename Result>
struct call_with_retries_op
    : enable_shared_from_this<call_with_retries_op<BalancingCallOp, Handler, Interpreter, Result>>
{
    using yield_context_type = yplatform::yield_context<call_with_retries_op>;
    using balancing_call_op_ptr = shared_ptr<BalancingCallOp>;
    using handler_type = std::decay_t<Handler>;
    using interpreter_type = std::decay_t<Interpreter>;
    using result_type = std::decay_t<Result>;

    template <typename... Args>
    static void run(boost::asio::io_service& io, Args&&... args)
    {
        auto op = std::make_shared<call_with_retries_op>(io, std::forward<Args>(args)...);
        yplatform::spawn(io, op);
    }

    call_with_retries_op(
        boost::asio::io_service& io,
        shared_ptr<settings> settings,
        shared_ptr<retry_settings> retry_settings,
        detail::request_stat_ptr stat,
        balancing_call_op_ptr call_op,
        const boost::optional<detail::backoff>& backoff,
        task_context_ptr ctx,
        request req,
        const options& initial_opt,
        boost::optional<unsigned> max_attempts,
        interpreter_type interpreter,
        handler_type handler)
        : timer(io)
        , http_settings(settings)
        , retry_settings(retry_settings)
        , stat(stat)
        , call_op(call_op)
        , ctx(ctx)
        , req(std::move(req))
        , opt(initial_opt)
        , max_attempts(calc_max_attempts(max_attempts, retry_settings))
        , interpreter(std::move(interpreter))
        , handler(std::move(handler))
        , backoff(backoff)
    {
        opt.timeouts = eval_timeouts(ctx, opt, *http_settings);
        deadline_total = time_traits::clock::now() + opt.timeouts.total;
        if (retry_settings->split_timeout)
        {
            opt.timeouts.total /= 2;
        }
        req.attempt = 0;
    }

    void operator()(yield_context_type yield_context)
    {
        try
        {
            reenter(yield_context)
            {
                correct_timeout();
                yield make_request(yield_context.capture(ec, response, cont));
                req.attempt++;
                if (ec)
                {
                    if (!retriable(ec))
                    {
                        // count perm error as success, because it was clients fault, no reason to
                        // retry it or change wrs stats
                        count_successfull_request();
                        return handler(ec, {});
                    }

                    count_failed_request();
                    if (can_retry(has_noretry_header))
                    {
                        return schedule_retry(yield_context);
                    }
                    return yplatform::safe_call(ctx, handler, ec, result_type());
                }

                has_noretry_header = response.headers.contains("x-request-noretry");
                if (code_needs_retry(response.status))
                {
                    count_failed_request();
                    if (can_retry(has_noretry_header))
                    {
                        return schedule_retry(yield_context);
                    }
                    yield interpreter(
                        std::move(response), yield_context.capture(response_status, result));
                    return yplatform::safe_call(
                        ctx,
                        handler,
                        response_status_to_error_code(response_status),
                        std::move(result));
                }

                yield interpreter(
                    std::move(response), yield_context.capture(response_status, result));
                if (response_status == response_status::tmp_error)
                {
                    count_failed_request();
                    if (can_retry(has_noretry_header))
                    {
                        return schedule_retry(yield_context);
                    }
                    return yplatform::safe_call(
                        ctx, handler, http_error::bad_response, std::move(result));
                }

                // count perm error as success, because it was clients fault, no reason to
                // retry it or change wrs stats
                count_successfull_request();
                yplatform::safe_call(
                    ctx,
                    handler,
                    response_status_to_error_code(response_status),
                    std::move(result));
            }
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_LOCAL(ctx, error)
                << "exception in call_with_retries_op: exception=\"" << e.what() << "\"";
            yplatform::safe_call(ctx, handler, http_error::unknown_error, std::move(result));
        }
    }

    unsigned calc_max_attempts(
        boost::optional<unsigned> max_attempts,
        shared_ptr<retry_settings> retry_settings)
    {
        return std::max(1U, max_attempts ? *max_attempts : retry_settings->max_attempts);
    }

    void correct_timeout()
    {
        if (req.attempt == 0) return; // calc for retries only

        auto estimate =
            std::max(deadline_total - time_traits::clock::now(), time_traits::duration::zero());
        auto attempts = std::max<int>(1, max_attempts - req.attempt);
        opt.timeouts.total = retry_settings->split_timeout ? estimate / attempts : estimate;
    }

    template <typename ResponseHandler>
    void make_request(ResponseHandler&& handler)
    {
        (*call_op)(ctx, req, opt, cont, std::forward<ResponseHandler>(handler));
    }

    error_code response_status_to_error_code(response_status status)
    {
        return status == response_status::ok ? http_error::success : http_error::bad_response;
    }

    void count_failed_request()
    {
        stat->count_failed_request(ec, req.attempt - 1, cont->last_selected_node);
    }

    void count_successfull_request()
    {
        stat->count_successfull_request(req.attempt - 1, cont->last_selected_node);
    }

    void schedule_retry(yield_context_type yield_context)
    {
        if (backoff)
        {
            auto self = yplatform::shared_from(this);
            wait(
                backoff->calc_delay(ctx, req.attempt),
                [yield_context, this, self](error_code ec) mutable {
                    if (ec) return handler(ec, {});
                    yield_context.respawn()();
                });
        }
        else
        {
            yield_context.respawn()();
        }
    }

    bool code_needs_retry(int code)
    {
        return retry_settings->codes.contains(code);
    }

    bool can_retry(bool has_noretry_header)
    {
        if (req.attempt >= max_attempts) return false;
        if (has_noretry_header)
        {
            YLOG_CTX_LOCAL(ctx, info) << "can't retry - got x-request-noretry header";
            return false;
        }
        if (retry_budget_exceeded())
        {
            YLOG_CTX_LOCAL(ctx, info) << "can't retry - retry budget exceeded";
            return false;
        }
        if (time_traits::clock::now() + http_settings->min_request_duration >= ctx->deadline())
        {
            YLOG_CTX_LOCAL(ctx, info) << "can't retry - too little time before deadline";
            return false;
        }
        return true;
    }

    bool retry_budget_exceeded()
    {
        if (!retry_settings->budget.enabled) return false;
        return stat->calc_retries_ratio() >= retry_settings->budget.value;
    }

    template <typename WaitHandler>
    void wait(time_traits::duration delay, WaitHandler&& handler)
    {
        timer.expires_from_now(delay);
        timer.async_wait(std::forward<WaitHandler>(handler));
    }

    time_traits::timer timer;
    shared_ptr<settings> http_settings;
    shared_ptr<retry_settings> retry_settings;
    detail::request_stat_ptr stat;
    balancing_call_op_ptr call_op;
    task_context_ptr ctx;
    request req;
    options opt;
    unsigned const max_attempts;
    interpreter_type interpreter;
    handler_type handler;
    boost::optional<detail::backoff> backoff;
    time_traits::time_point deadline_total;

    boost::system::error_code ec;
    response response;
    detail::continuation_ptr cont;
    bool has_noretry_header = false;
    response_status response_status = response_status::ok;
    result_type result;
};

}

#include <yplatform/unyield.h>
