#pragma once

#include <ymod_httpclient/types.h>
#include <yplatform/exception.h>

#include <boost/tuple/tuple.hpp>
#include <boost/exception/all.hpp>
#include <vector>

namespace ymod_httpclient {
using boost::system::error_code;

namespace http_error {

enum code : int
{
    success = 0,
    connect_error,
    ssl_error,
    read_error,
    write_error,
    server_response_error,
    server_status_error,
    server_header_error,
    response_handler_error,
    session_closed_error,
    parse_response_error,
    invalid_url,
    unknown_error,
    task_canceled,
    connection_timeout,
    request_timeout,
    eof_error,
    protocol_error,
    unsupported_scheme,
    request_uri_too_long,
    no_service_ticket,
    task_throttled,
    bad_response,

    COUNT
    // DO NOT FORGET TO EXTEND error_names
};

inline const string& message(code err)
{
    static constexpr std::initializer_list<const char*> names = {
        "success",
        "connect error",
        "ssl handshake error",
        "read error",
        "write error",
        "bad server response",
        "server returned bad status",
        "server returned bad headers",
        "response handler exception",
        "session closed",
        "parse response error",
        "invalid_url",
        "unknown_error",
        "task_canceled",
        "connection_timeout",
        "request_timeout",
        "unexpected eof",
        "protocol error",
        "unsupported scheme",
        "request uri too long",
        "no service ticket",
        "task throttled",
        "bad response",
    };
    static_assert(
        names.size() == code::COUNT, "Error codes count doesn't correspond with error names count");
    static const std::vector<string> error_names(names.begin(), names.end());

    return error_names[err];
}

class error_category : public boost::system::error_category
{
public:
    const char* name() const noexcept
    {
        return "ymod_httpclient";
    }

    string message(int ev) const
    {
        return http_error::message(code(ev));
    }

    static const error_category& instance()
    {
        static const error_category category;
        return category;
    }

private:
    error_category() = default;
};

inline boost::system::error_code make_error_code(code error)
{
    return boost::system::error_code(static_cast<int>(error), error_category::instance());
}

}

using errc = http_error::code;

class error : public std::runtime_error
{
public:
    explicit inline error(const string& message) : std::runtime_error(message)
    {
    }

    virtual std::exception_ptr exception_ptr() const = 0;
};

#define ERROR_DEF(name, xmessage)                                                                  \
public                                                                                             \
    error{ explicit inline name() : error(xmessage){} inline name(const string& msg) :             \
               error(msg){} virtual inline ~name(){} virtual std::exception_ptr exception_ptr()    \
                   const override{ return std::make_exception_ptr(*this);                          \
    }                                                                                              \
    }                                                                                              \
    ;

struct unknown_error : ERROR_DEF(unknown_error, "ymod_httpclient unknown error");

struct enqueue_error : ERROR_DEF(enqueue_error, "enqueue error");

struct resolve_error : ERROR_DEF(resolve_error, "resolve error");

struct connect_error : ERROR_DEF(connect_error, "connect error");

struct connection_timeout : ERROR_DEF(connection_timeout, "connection timeout");

struct ssl_error : ERROR_DEF(ssl_error, "ssl handshake error");

struct read_error : ERROR_DEF(read_error, "read error");

struct write_error : ERROR_DEF(write_error, "write error");

struct resp_size_error : ERROR_DEF(resp_size_error, "response size error");

// deprecated error:
struct xml_parse_error : ERROR_DEF(xml_parse_error, "cannot parse xml response");

struct data_parse_error : ERROR_DEF(data_parse_error, "cannot parse response");

struct bad_url_error : ERROR_DEF(bad_url_error, "bad URL");

struct server_response_error : ERROR_DEF(server_response_error, "bad server response");

struct server_status_error : ERROR_DEF(server_status_error, "server returned bad status");

struct server_header_error : ERROR_DEF(server_header_error, "server returned bad headers");

struct task_canceled_error : ERROR_DEF(task_canceled_error, "task canceled");

struct request_timeout_error : ERROR_DEF(request_timeout_error, "request timeout");

struct parse_response_error : ERROR_DEF(parse_response_error, "parse response error");

struct eof_error : ERROR_DEF(eof_error, "unexpected eof");

struct request_uri_too_long_error : ERROR_DEF(request_uri_too_long_error, "request uri too long");

struct no_service_ticket_error : ERROR_DEF(no_service_ticket_error, "no service ticket");

struct task_throttled_error : ERROR_DEF(task_throttled_error, "task throttled");

#undef ERROR_DEF
}

namespace boost { namespace system {

template <>
struct is_error_code_enum<ymod_httpclient::http_error::code>
{
    static const bool value = true;
};

}}

namespace yhttp {

using ymod_httpclient::errc;

}
