#pragma once

#include <ymod_httpclient/multipart.h>
#include <ymod_httpclient/types.h>
#include <yplatform/time_traits.h>
#include <boost/optional.hpp>
#include <variant>

namespace ymod_httpclient {

struct timeouts
{
    timeouts() : connect(time_traits::duration::max()), total(time_traits::duration::max())
    {
    }

    time_traits::duration connect;
    time_traits::duration total;
};

struct remote_point_info
{
    string proto;
    string host;
    unsigned short port;
    string uri_prefix;
    timeouts operation_timeouts;
    bool reuse_connection;
};

typedef boost::shared_ptr<remote_point_info> remote_point_info_ptr;

struct options
{
    boost::optional<bool> log_post_body;
    boost::optional<bool> log_headers;
    boost::optional<bool> reuse_connection;
    ::ymod_httpclient::timeouts timeouts;
};

struct typed_client_options : options
{
    boost::optional<unsigned> max_attempts;
};

using cluster_client_options = typed_client_options;

struct headers_param;

class request
{
public:
    enum class method_t
    {
        GET,
        HEAD,
        POST,
        PUT,
        DELETE
    };

    using headers_type = std::variant<string, header_dict>;

    static request POST(string url, string&& body);
    static request POST(string url, headers_param headers, string&& body);
    static request MPOST(string url, multipart&& body);
    static request MPOST(string url, headers_param headers, multipart&& body);
    static request PUT(string url, string&& body);
    static request PUT(string url, headers_param headers, string&& body);
    static request GET(string url);
    static request GET(string url, headers_param headers);
    static request HEAD(string url);
    static request HEAD(string url, headers_param headers);
    static request DELETE(string url);
    static request DELETE(string url, headers_param headers);

    method_t method;
    string url;
    headers_type headers;
    boost::shared_ptr<string> body;
    boost::shared_ptr<multipart> multipart_body;

    unsigned attempt = 0;
};

// Helper class to construct request::headers_type from various types.
struct headers_param
{
    headers_param(const char* headers) : data(headers)
    {
    }

    headers_param(string headers) : data(std::move(headers))
    {
    }

    headers_param(header_dict headers) : data(std::move(headers))
    {
    }

    headers_param(std::initializer_list<std::pair<string, string>> headers)
        : data(header_dict(headers.begin(), headers.end()))
    {
    }

    request::headers_type data;
};

void add(request::headers_type& headers, const string& key, const string& value);

std::ostream& operator<<(std::ostream& os, const request::headers_type& headers);

}

namespace yhttp {

using ymod_httpclient::request;
using ymod_httpclient::options;

}

#include <ymod_httpclient/impl/request.ipp>
