#pragma once

#include <ymod_httpclient/types.h>
#include <yplatform/net/dns/resolver_settings.h>
#include <yplatform/net/settings.h>
#include <yplatform/net/ssl_settings.h>
#include <yplatform/ptree.h>
#include <yplatform/find.h>
#include <vector>
#include <unordered_set>
#include <climits>

namespace ymod_httpclient {

struct settings
{
    string user_agent;
    bool ssl_verify_hostname = true;
    bool enable_logging = false;
    bool enable_post_args_logging = false;
    size_t post_args_log_entry_max_size = 1000;
    bool enable_headers_logging = false;
    std::unordered_set<std::string> protected_log_headers = { "x-ya-service-ticket",
                                                              "x-ya-user-ticket",
                                                              "cookie",
                                                              "authorization" };
    bool reuse_connection = false;
    unsigned int preffered_pool_size = 100U;
    unsigned int max_request_line_size = std::numeric_limits<unsigned int>::max();
    time_traits::duration poll_timeout = time_traits::duration::max();
    time_traits::duration connect_attempt_timeout = time_traits::milliseconds(500);
    time_traits::duration default_request_timeout = time_traits::seconds(15);
    time_traits::duration min_request_duration = time_traits::microseconds(100);
    time_traits::duration reactor_overload_delay = time_traits::duration::max();
    yplatform::net::client_settings socket;
    yplatform::net::ssl_settings ssl;
    yplatform::net::dns::resolver_service_settings dns;
    bool send_ssl_server_name = true;
    struct
    {
        string module;
        bool ignore_errors = false;
    } service_ticket_provider;

    boost::optional<yplatform::log::typed::logger> tskv_logger;
    std::string name_of_uniq_id_field = "y_context";
    std::string name_of_request_id_field = "";

    settings()
    {
        socket.resolve_timeout = time_traits::milliseconds(500);
        socket.connect_timeout = time_traits::milliseconds(500);
        dns.cache_ttl = time_traits::seconds(60);
    }

    void parse_ptree(const yplatform::ptree& data)
    {
        // check deprecated options
        if (data.count("default"))
            throw std::domain_error("config section 'default' is deprecated, "
                                    "please rename it to 'socket'");
        if (data.count("net_settings"))
            throw std::domain_error("config section 'net_settings' is deprecated, "
                                    "please rename it to 'socket'");
        if (data.count("ssl_context"))
            throw std::domain_error("config section 'ssl_context' is deprecated, "
                                    "please rename it to 'ssl'");
        if (data.count("ssl_verify_hostname"))
            throw std::domain_error("ssl_verify_hostname option is deprecated, "
                                    "please use ssl.verify_hostname");
        if (data.count("profile_logging"))
            throw std::domain_error("profile_logging option is deprecated, "
                                    "please remove it");
        if (data.count("debug_logging"))
            throw std::domain_error("debug_logging option is deprecated, "
                                    "please remove it");

        user_agent = data.get("user_agent", user_agent);
        // This allows user to write all ssl settings in a single section.
        ssl_verify_hostname = data.get("ssl.verify_hostname", ssl_verify_hostname);
        enable_logging = data.get("enable_logging", enable_logging);
        auto tskv = data.get("tskv_logging", false);
        if (tskv)
        {
            tskv_logger = yplatform::find<yplatform::log::typed::logger>(
                data.get("tskv_log_name", "HttpTskv"));
        }
        enable_post_args_logging = data.get("post_args_logging", enable_post_args_logging);
        post_args_log_entry_max_size =
            data.get("post_args_log_entry_max_size", post_args_log_entry_max_size);
        enable_headers_logging = data.get<bool>("headers_logging", enable_headers_logging);
        fill_protected_log_headers(data);
        reuse_connection = data.get<bool>("reuse_connection", reuse_connection);
        preffered_pool_size = data.get("preffered_pool_size", preffered_pool_size);
        max_request_line_size = data.get("max_request_line_size", max_request_line_size);
        poll_timeout = data.get("poll_timeout", poll_timeout);
        connect_attempt_timeout = data.get("connect_attempt_timeout", connect_attempt_timeout);
        default_request_timeout = data.get("default_request_timeout", default_request_timeout);
        min_request_duration = data.get("min_request_duration", min_request_duration);

        auto congestion_control = data.get_child_optional("congestion_control");
        if (congestion_control)
        {
            reactor_overload_delay =
                congestion_control->get("reactor_overload_delay", reactor_overload_delay);
        }

        socket.tcp_no_delay = true; // force TCP NODELAY by default
        socket.parse_ptree(data.get_child("socket", yplatform::ptree()));
        ssl.parse_ptree(data.get_child("ssl", yplatform::ptree()));
        dns.parse(data.get_child("dns", yplatform::ptree()));

        send_ssl_server_name = data.get("send_ssl_server_name", send_ssl_server_name);
        name_of_uniq_id_field = data.get("name_of_uniq_id_field", name_of_uniq_id_field);
        name_of_request_id_field = data.get("name_of_request_id_field", name_of_request_id_field);

        auto& [module, ignore_errors] = service_ticket_provider;
        module = data.get("service_ticket_provider.module", module);
        ignore_errors = data.get("service_ticket_provider.ignore_errors", ignore_errors);
    }

    void fill_protected_log_headers(const yplatform::ptree& data)
    {
        if (!data.count("protected_log_headers")) return;
        std::vector<std::string> headers;
        yplatform::read_ptree(headers, data, "protected_log_headers");
        protected_log_headers.clear();
        for (auto&& header : headers)
        {
            protected_log_headers.emplace(boost::to_lower_copy(header));
        }
    }
};

struct balancing_settings
{
    std::vector<string> nodes;

    enum
    {
        linear = 0,
        weighted_random
    } select_strategy = linear;

    struct wrs_settings
    {
        double factor = 2.0; // increase for more throttling
        bool ignore_last_failed = false;
        double max_initial_weight = 0.001;
        double max_weight_growth = 1.4;
        double momentary_fall_threshold = 1.2;
    } wrs;

    void parse_ptree(const yplatform::ptree& conf)
    {
        for (auto& node : boost::make_iterator_range(conf.equal_range("hosts")))
        {
            nodes.push_back(node.second.data());
        }
        // alias for hosts
        for (auto& node : boost::make_iterator_range(conf.equal_range("nodes")))
        {
            nodes.push_back(node.second.data());
        }

        auto strategy = conf.get("select_strategy", std::string{});
        if (strategy.empty() || strategy == "linear")
        {
            select_strategy = balancing_settings::linear;
        }
        else if (strategy == "weighted_random")
        {
            select_strategy = balancing_settings::weighted_random;
            wrs.factor = conf.get<double>("wrs_factor", wrs.factor);
            wrs.factor = conf.get<double>("wrs.factor", wrs.factor);
            wrs.max_initial_weight =
                conf.get<double>("wrs.max_initial_weight", wrs.max_initial_weight);
            wrs.max_weight_growth =
                conf.get<double>("wrs.max_weight_growth", wrs.max_weight_growth);
            wrs.momentary_fall_threshold =
                conf.get<double>("wrs.momentary_fall_threshold", wrs.momentary_fall_threshold);
            wrs.ignore_last_failed =
                conf.get<bool>("wrs.ignore_last_failed", wrs.ignore_last_failed);
        }
        else
        {
            throw std::runtime_error("unknown strategy \"" + strategy + "\"");
        }
    }
};

struct backoff_settings
{
    time_traits::duration base = time_traits::milliseconds(100);
    time_traits::duration max = time_traits::milliseconds(1000);
    double multiplier = 2.0;

    void parse_ptree(const yplatform::ptree& conf)
    {
        base = conf.get<time_traits::duration>("base");
        if (base <= time_traits::duration::zero())
            throw std::runtime_error("backoff base duration should be above zero");

        max = conf.get<time_traits::duration>("max");
        if (max < base)
            throw std::runtime_error("backoff max duration should be not less than base duration");

        multiplier = conf.get<double>("multiplier");
        if (multiplier < 1.0) throw std::runtime_error("backoff multiplier should be above 1.0");
    }
};

struct retry_settings
{
    unsigned max_attempts = 1;
    std::set<int> codes = { 500, 502, 503, 504 }; // retriable codes
    bool split_timeout = false;

    struct
    {
        bool enabled = false;
        double value = 0.2;
        bool count_connect_errors = false;
    } budget;

    boost::optional<backoff_settings> backoff;

    void parse_ptree(const yplatform::ptree& conf)
    {
        max_attempts = conf.get("max_attempts", max_attempts);
        if (conf.find("codes") != conf.not_found())
        {
            codes.clear();
            yplatform::read_ptree(codes, conf, "codes");
        }

        split_timeout = conf.get("split_timeout", split_timeout);
        auto budget_conf = conf.get_child_optional("budget");
        if (budget_conf)
        {
            budget.enabled = budget_conf->get("enabled", true);
            budget.value = budget_conf->get<double>("value");
            budget.count_connect_errors =
                budget_conf->get("count_connect_errors", budget.count_connect_errors);
        }
        auto backoff_conf = conf.get_child_optional("backoff");
        if (backoff_conf && backoff_conf->get("enabled", true))
        {
            backoff_settings settings;
            settings.parse_ptree(*backoff_conf);
            backoff = settings;
        }
    }
};
}
