#pragma once

#include <ymod_httpclient/detail/balancing_call_op.h>
#include <ymod_httpclient/detail/call_with_retries_op.h>
#include <ymod_httpclient/call.h>
#include <ymod_httpclient/settings.h>
#include <yplatform/module.h>

namespace ymod_httpclient {

template <typename Result>
class typed_client : public yplatform::module
{
    using result_type = std::decay_t<Result>;

public:
    struct settings
        : ::ymod_httpclient::settings
        , balancing_settings
    {
        unsigned stats_period = 5; // stats collecting period in seconds

        retry_settings retry_policy;

        void parse_ptree(const yplatform::ptree& conf)
        {
            ymod_httpclient::settings::parse_ptree(conf);
            balancing_settings::parse_ptree(conf);

            stats_period = conf.get("stats_period", stats_period);
            auto retry_policy_conf = conf.get_child_optional("retry_policy");
            if (retry_policy_conf)
            {
                retry_policy.parse_ptree(*retry_policy_conf);
            }
        }

        static settings from_ptree(const yplatform::ptree& conf)
        {
            settings res;
            res.parse_ptree(conf);
            return res;
        }
    };

    typed_client(yplatform::reactor_ptr reactor, const settings& st)
        : io(*reactor->io())
        , st(make_shared<settings>(st))
        , stat(make_shared<detail::request_stat>(
              st.stats_period,
              st.retry_policy.budget.count_connect_errors,
              st.nodes.size()))
        , balancing_call_op(
              std::make_shared<detail::balancing_call_op>(io, reactor, this->st, this->st, stat))
    {
    }

    typed_client(boost::asio::io_service& io, const settings& st)
        : typed_client(boost::make_shared<yplatform::reactor>(io, 1), st)
    {
    }

    typed_client(yplatform::reactor& reactor, const yplatform::ptree& conf)
        : typed_client(
              yplatform::reactor::make_not_owning_copy(reactor),
              settings::from_ptree(conf))
    {
    }

    void logger(const yplatform::log::source& logger) // shadows base class method
    {
        contains_logger::logger(logger);
        balancing_call_op->logger(logger);
    }

    void name(const string& name) // shadows base class method
    {
        yplatform::module::name(name);
        balancing_call_op->name(name);
    }

    template <typename Interpreter, typename CompletionToken>
    auto async_run(
        task_context_ptr ctx,
        request req,
        const typed_client_options& options,
        Interpreter&& interpreter,
        CompletionToken&& token)
    {
        using Signature = void(boost::system::error_code, result_type);
        boost::asio::async_completion<CompletionToken, Signature> init(token);
        run_request(
            ctx,
            std::move(req),
            options,
            std::forward<Interpreter>(interpreter),
            [handler = init.completion_handler](
                boost::system::error_code ec, result_type response) mutable {
                boost::asio::dispatch(detail::bind(std::move(handler), ec, std::move(response)));
            });
        return init.result.get();
    }

    template <typename Interpreter, typename CompletionToken>
    auto async_run(
        task_context_ptr ctx,
        request req,
        Interpreter&& interpreter,
        CompletionToken&& token)
    {
        return async_run(
            ctx,
            std::move(req),
            {},
            std::forward<Interpreter>(interpreter),
            std::forward<CompletionToken>(token));
    }

    yplatform::ptree get_stats() const
    {
        yplatform::ptree ret = balancing_call_op->get_stats();
        if (st->retry_policy.budget.enabled)
        {
            ret.put("retries.budget", std::to_string(st->retry_policy.budget.value));
            ret.put(
                "retries.ratio",
                std::to_string(stat->retries_ratio)); // dirty read without synchronization
        }
        return ret;
    }

private:
    template <typename Interpreter, typename Handler>
    void run_request(
        task_context_ptr ctx,
        request req,
        const typed_client_options& options,
        Interpreter&& interpreter,
        Handler&& handler)
    {
        using interpreter_type = std::decay_t<Interpreter>;
        using call_with_retries_op = detail::
            call_with_retries_op<detail::balancing_call_op, Handler, interpreter_type, result_type>;
        call_with_retries_op::run(
            io,
            st,
            shared_ptr<retry_settings>(st, &st->retry_policy),
            stat,
            balancing_call_op,
            backoff,
            ctx,
            std::move(req),
            options,
            options.max_attempts,
            std::forward<Interpreter>(interpreter),
            std::forward<Handler>(handler));
    }

    boost::asio::io_service& io;
    shared_ptr<settings> st;
    shared_ptr<detail::request_stat> stat;
    shared_ptr<detail::balancing_call_op> balancing_call_op;
    boost::optional<detail::backoff> backoff;
};

}

namespace yhttp {

template <typename ResultT>
using typed_client = ymod_httpclient::typed_client<ResultT>;

}
