#pragma once

#include <ymod_httpclient/detail/balancing_call_op.h>
#include <yplatform/algorithm/weighted_random_select.h>
#include <yplatform/util/safe_call.h>

namespace ymod_httpclient::detail {

namespace ph = std::placeholders;

template <typename CallType>
struct balancing_call_op_impl
    : public std::enable_shared_from_this<balancing_call_op_impl<CallType>>
{
    using call_ptr = shared_ptr<CallType>;

    balancing_call_op_impl(
        boost::asio::io_service& io,
        shared_ptr<balancing_settings> settings,
        request_stat_ptr stat,
        shared_ptr<CallType> call)
        : io(io), stat(stat), call(call), settings(settings)
    {
        if (settings->nodes.empty()) throw std::runtime_error("no nodes for cluster_client");
        if (settings->nodes.size() == 1) settings->select_strategy = balancing_settings::linear;
        if (settings->select_strategy == balancing_settings::weighted_random)
        {
            if (settings->wrs.max_initial_weight <= 0.0)
            {
                throw std::runtime_error("max_initial_weight must be greater than 0.0");
            }
            if (settings->wrs.max_weight_growth <= 1.0)
            {
                throw std::runtime_error("max_weight_growth must be greater than 1.0");
            }
            if (settings->wrs.momentary_fall_threshold <= 1.0)
            {
                throw std::runtime_error("momentary_fall_threshold must be greater than 1.0");
            }
            wrs = make_shared<yplatform::weighted_random_select>();
        }
    }

    template <typename Handler>
    void operator()(
        task_context_ptr ctx,
        request req,
        const options& options,
        continuation_ptr cont,
        Handler&& handler)
    {
        if (!cont)
        {
            cont = make_shared<continuation_token>();
        }
        auto self = yplatform::shared_from(this);
        io.post([ctx,
                 req = std::move(req),
                 options,
                 cont,
                 h = std::forward<Handler>(handler),
                 this,
                 self]() mutable {
            cont->last_selected_node = select_node(cont->last_selected_node, req.attempt);
            req.url = settings->nodes[cont->last_selected_node] + req.url;
            call->async_run(
                ctx, std::move(req), options, std::bind(std::move(h), ph::_1, ph::_2, cont));
        });
    }

    size_t select_node(unsigned last_selected_node, unsigned attempt)
    {
        if (settings->select_strategy == balancing_settings::weighted_random)
        {
            auto corrected_factor = settings->wrs.factor * (attempt + 1) * (attempt + 1);
            std::vector<double> weights_(settings->nodes.size());
            calc_weights(
                stat->wrs, weights_, stat->weights_average, corrected_factor, settings->wrs);
            if (attempt > 0 && settings->wrs.ignore_last_failed)
            {
                weights_[last_selected_node] = 0;
            }
            return (*wrs)(weights_);
        }
        else
        {
            return attempt % settings->nodes.size();
        }
    }

    yplatform::ptree get_stats()
    {
        yplatform::ptree ret = call->get_stats();
        if (wrs)
        {
            yplatform::ptree weights_ptree;
            std::vector<double> weights;
            // dirty read without synchronization
            calc_weights(
                stat->wrs, weights, stat->weights_average, settings->wrs.factor, settings->wrs);
            double weights_sum = std::accumulate(weights.begin(), weights.end(), 0.0);
            if (weights.size() == settings->nodes.size())
            {
                for (auto i = 0UL; i < weights.size(); ++i)
                {
                    auto node = settings->nodes[i]; // assume a strict order
                    std::replace_if(
                        node.begin(), node.end(), [](char ch) { return !isalnum(ch); }, '_');
                    weights_ptree.put(node, std::to_string(weights[i] / weights_sum));
                }
                ret.add_child("wrs_weights", weights_ptree);
            }
        }
        return ret;
    }

    boost::asio::io_service& io;
    request_stat_ptr stat;
    call_ptr call;
    shared_ptr<balancing_settings> settings;
    shared_ptr<yplatform::weighted_random_select> wrs;
};

}
