#include <ymod_httpclient/h2/client.h>

#include "connection_control_coroutine.h"
#include "write_coroutine.h"
#include <ymod_httpclient/util/url_parser.h>
#include <yplatform/net/io_data.h>
#include <yplatform/util/safe_call.h>

namespace ymod_httpclient { namespace h2 {

class client::impl : public yplatform::log::contains_logger
{
public:
    impl(boost::asio::io_service& io, const settings& st)
        : io_data(io), settings(st), pool(make_shared<session_pool>())
    {
        io_data.setup_ssl(settings.ssl);
        io_data.setup_dns(settings.dns);
        SSL_CTX_set_alpn_protos(
            io_data.get_ssl_context()->native_handle(), H2_PROTO_ALPN, H2_PROTO_ALPN_SZ);
    }

    // Client objects are not guaranteed to be shared pointers,
    // therefore we place this function in impl class to ensure
    // both thread safety (by posting to io service) and data lifetime guarantees.
    void async_run_impl(
        task_context_ptr ctx,
        request& req,
        const options& options,
        client::handler_type& handler)
    {
        auto req_data = make_shared<request_data>(*get_io());
        auto err =
            parse_url(req.url, req_data->scheme, req_data->host, req_data->port, req_data->path);
        if (err)
        {
            handler(err, response());
            return;
        }

        if (req_data->scheme.size() && req_data->scheme != "https")
        {
            handler(http_error::unsupported_scheme, response());
            return;
        }
        else
        {
            req_data->scheme = "https";
        }
        req_data->ctx = ctx;
        if (req_data->port == 0) req_data->port = 443;
        req_data->method = std::move(req.method);
        req_data->headers = std::move(req.headers);
        req_data->body = std::move(req.body);
        req_data->handler = std::move(handler);
        req_data->deadline = time_traits::clock::now() +
            (options.timeouts.total == time_traits::duration::max() ?
                 settings.default_request_timeout :
                 options.timeouts.total);

        if (ctx->deadline() != time_traits::time_point::max())
        {
            req_data->deadline = std::min(req_data->deadline, ctx->deadline());
        }

        shared_session session;

        bool created = false;
        // [XIVA-1869] Skip incorrect sessions.
        for (size_t i = 0; i < pool->size() && !session; ++i)
        {
            // Index might be invalid because of disconnects.
            if (next_pool_index >= pool->size())
            {
                next_pool_index %= pool->size();
            }
            auto next_session = (*pool)[next_pool_index++];
            if (!next_session->shutdown) session = next_session;
        }

        if (!session)
        {
            session = make_shared<h2::session>(
                io_data, req_data->host, req_data->port, settings, this->logger());
            created = true;
        }

        session->put_pending(req_data);
        start_pending_timer(session, req_data);

        if (created)
        {
            connection_control_coroutine coro_main(session, pool);
            coro_main();
        }
        else if (session->idle)
        {
            write_coroutine write_coro(session);
            write_coro();
        }
        else if (pool->size() < settings.max_pool_size)
        {
            // Lazily create more connections if client is under high load.
            auto excess_session = make_shared<h2::session>(
                io_data, req_data->host, req_data->port, settings, this->logger());
            connection_control_coroutine coro_main(excess_session, pool);
            coro_main();
        }
    }

    boost::asio::io_service* get_io()
    {
        return io_data.get_io();
    }

    void async_get_stats(const client::stats_handler& handler)
    {
        stats ret;
        ret.sessions.reserve(pool->size());
        for (size_t i = 0; i < pool->size(); ++i)
        {
            auto& session = (*pool)[i];
            ret.sessions.push_back(session->get_stats());
        }
        yplatform::safe_call(handler, ret);
    }

private:
    yplatform::net::io_data io_data;
    h2::settings settings;
    shared_ptr<session_pool> pool;
    size_t next_pool_index = 0;
};

client::client(boost::asio::io_service& io, const settings& st) : impl_(new client::impl(io, st))
{
}

void client::async_run(task_context_ptr ctx, request req, handler_type handler)
{
    async_run(ctx, std::move(req), options(), std::move(handler));
}

void client::async_run(
    task_context_ptr ctx,
    request req,
    const options& options,
    handler_type handler)
{
    impl_->get_io()->post(
        std::bind(&impl::async_run_impl, impl_, ctx, std::move(req), options, std::move(handler)));
}

void client::async_get_stats(stats_handler handler)
{
    impl_->get_io()->post(std::bind(&impl::async_get_stats, impl_, std::move(handler)));
}

}}
