#pragma once

#include "nghttp2_impl.h"
#include "session.h"
#include "write_coroutine.h"
#include <yplatform/time_traits.h>
#include <boost/asio.hpp>
#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>
#include <openssl/ssl.h>

namespace ymod_httpclient { namespace h2 {

// Can be started only by a client instance.
struct connection_control_coroutine : public boost::asio::coroutine
{
    shared_session session;
    shared_session_pool pool;
    yplatform::log::source& logger;

    connection_control_coroutine(const shared_session& session, const shared_session_pool& pool)
        : session(session), pool(pool), logger(session->logger)
    {
        pool->attach(session);
    }

    void operator()(const error_code_t& ec = error_code_t(), size_t bytes = 0)
    {
        using namespace boost::asio;

        try
        {
            reenter(this)
            {
                // Drop session on connect, TLS or setup error.
                yield connect();
                if (ec)
                {
                    YLOG(logger, info) << "connect failed: " << ec.message();
                    session->fin_all_requests(
                        is_err_correct_eof(ec) ? errc::eof_error : errc::connect_error);
                    yield break;
                }

                YLOG(logger, info)
                    << "connected at "
                    << time_traits::to_string(time_traits::clock::now() - session->inited_at);

                yield tls_handshake();
                if (ec)
                {
                    YLOG(logger, info) << "TLS handshake failed: " << ec.message();
                    // Generally it's not always ssl_error because we can't get
                    // different short read reasons here.
                    session->fin_all_requests(errc::ssl_error);
                    yield break;
                }
                if (!tls_h2_negotiated(*session) || !setup_nghttp2(*session))
                {
                    session->fin_all_requests(errc::protocol_error);
                    yield break;
                }

                YLOG(logger, info)
                    << "tls handshake finished at "
                    << time_traits::to_string(time_traits::clock::now() - session->inited_at);

                // Starts a write coroutine in parallel.
                send_requests();

                // Write coroutine will close session's socket.
                while (session->socket.is_open())
                {
                    yield read();
                    if (ec)
                    {
                        if (ec == boost::asio::error::operation_aborted)
                        {
                            // Possible aborted by read timer, write timer
                            // or idle timeout (if no active requests).
                            close(errc::session_closed_error);
                        }
                        else
                        {
                            if (want_io(*session, ec))
                            {
                                YLOG(logger, error) << "read error: " << ec.message();
                            }
                            close(is_err_correct_eof(ec) ? errc::eof_error : errc::read_error);
                        }
                        yield break;
                    }

                    if (!consume_data(bytes))
                    {
                        close(errc::protocol_error);
                        yield break;
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            try
            {
                YLOG(logger, error) << "control coro exception: " << e.what();
            }
            catch (...)
            {
            }
            try
            {
                session->socket.close();
            }
            catch (...)
            {
            }
        }

        if (is_complete())
        {
            pool->detach(session);
        }
    }

    void connect()
    {
        session->connect_op.perform(
            session->host,
            session->port,
            session->settings.connect_attempt_timeout,
            session->settings.connect_timeout,
            *this);
    }

    void tls_handshake()
    {
        session->socket.async_tls_handshake(
            decltype(session->socket)::handshake_type::client,
            session->settings.tls_timeout,
            *this);
    }

    void send_requests()
    {
        session->idle = true;
        write_coroutine write_coro(session);
        write_coro();
    }

    void read()
    {
        session->socket.async_read(
            session->asio_read_buffer, session->settings.read_timeout, *this);
    }

    bool consume_data(size_t bytes)
    {
        auto size_ret =
            nghttp2_session_mem_recv(session->nghttp2_native, session->read_buffer.data(), bytes);
        if (size_ret != static_cast<ssize_t>(bytes))
        {
            YLOG(logger, error) << "protocol error while read: " << nghttp2_strerror(size_ret);
        }
        return size_ret == static_cast<ssize_t>(bytes);
    }

    void close(errc err)
    {
        session->socket.shutdown(true);
        session->fin_all_requests(err, "session closed");
    }
};

}}
