#pragma once

#include "utils.h"
#include "session.h"
#include <nghttp2/nghttp2.h>

#define MAX_HEADERS_SIZE 64 * 1024 * 1024

namespace ymod_httpclient { namespace h2 {
namespace callbacks {

static bool expect_final_response(const request_data& req)
{
    return (req.response.status / 100 == 1);
}

static bool contain_trailers(const nghttp2_frame* frame)
{
    return frame->headers.cat == NGHTTP2_HCAT_HEADERS;
}

static bool end_of(const nghttp2_frame* frame)
{
    return frame->hd.flags & NGHTTP2_FLAG_END_STREAM;
}

static ssize_t on_data_read(
    nghttp2_session*,
    int32_t stream_id,
    uint8_t* buf,
    size_t length,
    uint32_t* data_flags,
    [[maybe_unused]] nghttp2_data_source* source,
    void* user_data)
{
    auto session = static_cast<h2::session*>(user_data);
    auto reqit = session->active_requests.find(stream_id);
    if (reqit == session->active_requests.end())
    {
        return 0;
    }
    auto& req = reqit->second;
    assert(req.get() == static_cast<request_data*>(source->ptr));
    auto actual_length = req->body.size() - req->body_offset;
    if (actual_length <= length)
    {
        *data_flags |= NGHTTP2_DATA_FLAG_EOF;
    }
    else
    {
        actual_length = length;
    }
    std::copy_n(utils::to_uint8(&req->body[0] + req->body_offset), actual_length, buf);
    req->body_offset += actual_length;
    return static_cast<ssize_t>(actual_length);
}

static int on_header(
    nghttp2_session*,
    const nghttp2_frame* frame,
    const uint8_t* name,
    size_t namelen,
    const uint8_t* value,
    size_t valuelen,
    uint8_t /*flags*/,
    void* user_data)
{
    auto session = static_cast<h2::session*>(user_data);
    auto reqit = session->active_requests.find(frame->hd.stream_id);
    if (reqit == session->active_requests.end())
    {
        return 0;
    }
    auto& req = reqit->second;
    string name_string;

    switch (frame->hd.type)
    {
    case NGHTTP2_HEADERS:
        // ignore trailers
        if (contain_trailers(frame) && !expect_final_response(*req))
        {
            break;
        }

        name_string = utils::to_string(name, namelen);
        if (name_string == ":status")
        {
            req->response.status = std::stoul(utils::to_string(value, valuelen));
        }
        else
        {
            if (req->headers_size > MAX_HEADERS_SIZE)
            {
                nghttp2_submit_rst_stream(
                    session->nghttp2_native,
                    NGHTTP2_FLAG_NONE,
                    frame->hd.stream_id,
                    NGHTTP2_INTERNAL_ERROR);
                break;
            }
            req->headers_size += namelen += valuelen;

            if (name_string == "content-length")
            {
                req->content_length = std::stoul(utils::to_string(value, valuelen));
                req->response.body.reserve(req->content_length);
            }

            req->response.headers.emplace(name_string, utils::to_string(value, valuelen));
        }
        break;

    default:
        break;
    }
    return 0;
}

static int on_data_chunk(
    nghttp2_session*,
    uint8_t /*flags*/,
    int32_t stream_id,
    const uint8_t* data,
    size_t len,
    void* user_data)
{
    auto session = static_cast<h2::session*>(user_data);
    auto reqit = session->active_requests.find(stream_id);
    if (reqit != session->active_requests.end())
    {
        auto& req = reqit->second;
        req->response.body.append(reinterpret_cast<const char*>(data), len);
    }
    return 0;
}

int on_frame_recv(nghttp2_session*, const nghttp2_frame* frame, void* user_data)
{
    auto session = static_cast<h2::session*>(user_data);
    if (frame->hd.type == NGHTTP2_SETTINGS)
    {
        for (size_t i = 0; i < frame->settings.niv; ++i)
        {
            if (frame->settings.iv[i].settings_id == NGHTTP2_SETTINGS_MAX_CONCURRENT_STREAMS)
            {
                YLOG(session->logger, info)
                    << "server changed MAX_CONCURRENT_STREAMS to " << frame->settings.iv[i].value;
            }
        }
        return 0;
    }

    switch (frame->hd.type)
    {
    case NGHTTP2_SETTINGS:
        YLOG(session->logger, info) << "SETTINGS frame received";
        break;
    case NGHTTP2_PING:
        YLOG(session->logger, info) << "PING frame received";
        break;
    case NGHTTP2_GOAWAY:
        YLOG(session->logger, info) << "GOAWAY frame received";
        break;
    case NGHTTP2_WINDOW_UPDATE:
        YLOG(session->logger, info)
            << "WINDOW_UPDATE frame received sz=" << frame->window_update.window_size_increment
            << " reserved=" << frame->window_update.reserved;
        break;
    case NGHTTP2_ALTSVC:
        YLOG(session->logger, info) << "ALTSVC frame received";
        break;
    default:
        break;
    }

    auto reqit = session->active_requests.find(frame->hd.stream_id);
    if (reqit == session->active_requests.end())
    {
        return 0;
    }
    auto& req = reqit->second;

    switch (frame->hd.type)
    {
    case NGHTTP2_DATA:
    {
        if (end_of(frame))
        {
            session->fin_request(session->active_requests, reqit);
        }
        break;
    }
    case NGHTTP2_HEADERS:
        if (!expect_final_response(*req) && !contain_trailers(frame) && end_of(frame))
        {
            session->fin_request(session->active_requests, reqit);
        }
        break;

    default:
        break;
    }
    return 0;
}

static int on_frame_send(nghttp2_session*, const nghttp2_frame* frame, void* user_data)
{
    // Workaround for bug XIVA-1869.
    auto session = static_cast<h2::session*>(user_data);
    bool incorrect_frame = frame->hd.type == NGHTTP2_DATA && frame->hd.length == 0;
    if (session->active_requests.empty() && incorrect_frame)
    {
        YLOG(session->logger, warning)
            << "zero data frame while no active_requests "
            << " frame.stream_id=" << frame->hd.stream_id << " frame.type=" << frame->hd.type
            << " frame.flags=" << frame->hd.flags << " frame.length=" << frame->hd.length;
        // Close connection immediately.
        return NGHTTP2_ERR_CALLBACK_FAILURE;
    }

    if (!session->shutdown && incorrect_frame)
    {
        YLOG(session->logger, warning) << "incorrect frames are detected, preparing to shutdown";
        session->shutdown = true;
    }

    return 0;
}

static int on_stream_close(
    nghttp2_session*,
    int32_t stream_id,
    uint32_t error_code,
    void* user_data)
{
    auto session = static_cast<h2::session*>(user_data);
    session->fin_active_request(stream_id, errc::unknown_error, nghttp2_strerror(error_code));
    return 0;
}

} // namespace callbacks

static bool setup_nghttp2(session& session)
{
    nghttp2_session_callbacks* callbacks;
    int ret = nghttp2_session_callbacks_new(&callbacks);
    if (ret)
    {
        YLOG(session.logger, error) << "session setup failed: " << nghttp2_strerror(ret);
        return false;
    }
    std::unique_ptr<nghttp2_session_callbacks, decltype(&nghttp2_session_callbacks_del)>
        auto_deleter(callbacks, nghttp2_session_callbacks_del);

    nghttp2_session_callbacks_set_on_header_callback(callbacks, callbacks::on_header);
    nghttp2_session_callbacks_set_on_data_chunk_recv_callback(callbacks, callbacks::on_data_chunk);
    nghttp2_session_callbacks_set_on_frame_recv_callback(callbacks, callbacks::on_frame_recv);
    nghttp2_session_callbacks_set_on_frame_send_callback(callbacks, callbacks::on_frame_send);
    nghttp2_session_callbacks_set_on_stream_close_callback(callbacks, callbacks::on_stream_close);

    ret = nghttp2_session_client_new(&session.nghttp2_native, callbacks, &session);
    if (ret != 0)
    {
        YLOG(session.logger, error) << "session setup failed: " << nghttp2_strerror(ret);
        return false;
    }

    // From nghttp2 examples: typically client is just a *sink* and just process
    // data as much as possible. Use large window size by default.
    static const uint32_t window_size = 256 * 1024 * 1024;

    std::array<nghttp2_settings_entry, 3> iv{ { { NGHTTP2_SETTINGS_MAX_CONCURRENT_STREAMS, 100 },
                                                { NGHTTP2_SETTINGS_ENABLE_PUSH, 0 },
                                                { NGHTTP2_SETTINGS_INITIAL_WINDOW_SIZE,
                                                  window_size } } };
    ret = nghttp2_submit_settings(session.nghttp2_native, NGHTTP2_FLAG_NONE, iv.data(), iv.size());
    if (ret != 0)
    {
        YLOG(session.logger, error) << "session submit settings failed: " << nghttp2_strerror(ret);
        return false;
    }

    // Increase connection window size up to window_size
    ret = nghttp2_session_set_local_window_size(
        session.nghttp2_native, NGHTTP2_FLAG_NONE, 0, window_size);
    if (ret != 0)
    {
        YLOG(session.logger, error) << "session window setup failed: " << nghttp2_strerror(ret);
        return false;
    }

    return true;
}

static int32_t wrapped_nghttp2_submit_request(
    const shared_session& session,
    const shared_request_data& req)
{
    nghttp2_data_provider data_provider;
    data_provider.source.ptr = req.get();
    data_provider.read_callback = callbacks::on_data_read;

    auto nva = std::vector<nghttp2_nv>();
    nva.reserve(4 + req->headers.size());
    nva.push_back(utils::make_nv(":method", req->method));
    nva.push_back(utils::make_nv(":scheme", req->scheme));
    nva.push_back(utils::make_nv(":path", req->path));
    nva.push_back(utils::make_nv(":authority", req->host));
    for (auto& kv : req->headers)
    {
        nva.push_back(utils::make_nv(kv.first, kv.second, /*no index*/ true));
    }
    auto stream_id = nghttp2_submit_request(
        session->nghttp2_native, nullptr, nva.data(), nva.size(), &data_provider, session.get());
    return stream_id;
}

}}
