#pragma once

#include <ymod_httpclient/errors.h>
#include <yplatform/log.h>
#include <yplatform/time_traits.h>
#include <nghttp2/nghttp2.h>
#include <boost/asio.hpp>
#include <boost/noncopyable.hpp>
#include <functional>
#include <iostream>
#include <string>

namespace ymod_httpclient { namespace h2 {

namespace time_traits = time_traits;
using error_code_t = boost::system::error_code;
using handler_t = std::function<void(errc, response)>;

// Holds all connection data together to reduce memory allocations.
struct request_data : public boost::noncopyable
{
    request_data(boost::asio::io_service& io_service)
        : ctx(boost::make_shared<yplatform::task_context>())
        , inited_at(time_traits::clock::now())
        , wait_time(time_traits::duration::zero())
        , timer(io_service)
    {
    }

    ~request_data()
    {
        if (!finished())
        {
            YLOG_CTX_GLOBAL(ctx, error) << "request dropped";
            try
            {
                fin(errc::unknown_error);
            }
            catch (...)
            {
            }
        }
    }

    bool finished() const
    {
        return !handler;
    }

    void fin(errc err)
    {
        if (handler)
        {
            // TODO check err, status and content len and
            // log if no err and incomplete response.
            handler_t handler_copy;
            std::swap(handler_copy, handler);
            handler_copy(err, err ? ::yhttp::response() : std::move(response));
        }
    }

    task_context_ptr ctx;
    string method;
    string scheme;
    string path;
    string host;
    unsigned short port = 0;
    std::vector<std::pair<string, string>> headers;
    string body;
    size_t body_offset = 0;
    handler_t handler;
    time_traits::time_point inited_at;
    time_traits::duration wait_time;
    time_traits::time_point deadline;

    time_traits::timer timer;
    uint32_t stream_id = 0;
    size_t content_length = 0;
    size_t headers_size = 0; // for comparison with MAX_HEADERS_SIZE

    ::yhttp::response response;
};

using shared_request_data = shared_ptr<request_data>;

}}
