#pragma once

#include "protocol_constants.h"
#include "request_data.h"
#include "utils.h"
#include "write_buffer.h"
#include <ymod_httpclient/h2/client.h>
#include <ymod_httpclient/errors.h>
#include <yplatform/net/sequental_connect.h>
#include <yplatform/net/socket.h>
#include <yplatform/log.h>
#include <yplatform/time_traits.h>
#include <nghttp2/nghttp2.h>
#include <boost/asio.hpp>
#include <boost/noncopyable.hpp>
#include <functional>
#include <iostream>
#include <string>
#include <set>

namespace ymod_httpclient { namespace h2 {

namespace time_traits = time_traits;
using error_code_t = boost::system::error_code;

// Session is shared and served asynchronously by three parallel entities:
// connection coro, write coro and requests' timers.
// Session and coroutines are designed for single thread served I/O service.
struct session : public boost::noncopyable
{
    using socket_type = yplatform::net::tcp_socket;
    session(
        yplatform::net::io_data& io,
        string host,
        unsigned short port,
        const settings& st,
        const yplatform::log::source& parent_logger)
        : ctx(boost::make_shared<yplatform::task_context>())
        , settings(st)
        , socket(io)
        , connect_op(socket)
        , logger(parent_logger)
        , host(host)
        , port(port)
        , asio_read_buffer(read_buffer.data(), read_buffer.size())
        , inited_at(time_traits::clock::now())
    {
        logger.set_log_prefix(
            logger.get_log_prefix() + host + ":" + std::to_string(port) + ":" + ctx->uniq_id());
        connect_op.logger(logger);
    }

    ~session()
    {
        try
        {
            fin_all_requests(errc::session_closed_error, "session destroyed");
            nghttp2_session_del(nghttp2_native);
        }
        catch (...)
        {
        }
    }

    void put_pending(const shared_request_data& req)
    {
        YLOG_CTX(logger, req->ctx, debug) << "event=start path=" << req->path;
        pending_requests.push_back(req);
    }

    shared_request_data pop_pending()
    {
        if (pending_requests.empty())
        {
            throw std::runtime_error("attempt to pop element from an empty array");
        }
        auto req = pending_requests.front();
        req->timer.cancel();
        req->wait_time = time_traits::clock::now() - req->inited_at;
        pending_requests.pop_front();
        return req;
    }

    bool put_active(int32_t stream_id, const shared_request_data& req)
    {
        return active_requests.emplace(stream_id, req).second;
    }

    void fin_all_requests(errc err, const char* reason = nullptr)
    {
        while (pending_requests.size())
        {
            // Cleanup vector from the end.
            fin_request(pending_requests, std::prev(pending_requests.end()), err, reason);
        }
        while (active_requests.size())
        {
            fin_request(active_requests, active_requests.begin(), err, reason);
        }
    }

    // Finish detached request.
    void fin_request(const shared_request_data& req, errc err, const char* = nullptr)
    {
        if (!req->finished())
        {
            req->timer.cancel();
            req->timer.get_io_service().post(std::bind(&request_data::fin, req, err));
            auto total_time = time_traits::clock::now() - req->inited_at;
            YLOG_CTX(logger, req->ctx, info)
                << "event=fin tm={" << time_traits::to_string(req->wait_time) << ", "
                << time_traits::to_string(total_time) << "} "
                << " stream_id=" << req->stream_id << " status=" << req->response.status
                << (settings.log_request_path ? " path=" + req->path : string{});
        }
    }

    // Finish request in container.
    template <typename Container>
    void fin_request(
        Container& container,
        typename Container::iterator it,
        errc err = errc::success,
        const char* reason = nullptr)
    {
        if (it != container.end())
        {
            auto req = get_value(*it);
            container.erase(it);
            fin_request(req, err, reason);
        }
    }

    // Finish active request by stream id.
    void fin_active_request(int32_t stream_id, errc err, const char* reason = nullptr)
    {
        fin_request(active_requests, active_requests.find(stream_id), err, reason);
    }

    // Finish pending request by pointer.
    void fin_pending_request(const request_data* req, errc err, const char* reason = nullptr)
    {
        for (auto it = pending_requests.begin(); it != pending_requests.end(); ++it)
        {
            if (it->get() == req)
            {
                fin_request(pending_requests, it, err, reason);
                break;
            }
        }
    }

    session_stats get_stats() const
    {
        session_stats ret;
        ret.idle = idle;
        ret.pending_requests = pending_requests.size();
        ret.active_requests = active_requests.size();
        ret.local_port = socket.local_port();
        return ret;
    }

    task_context_ptr ctx;
    h2::settings settings;
    socket_type socket;
    yplatform::net::async_sequental_connect_op<socket_type> connect_op;
    yplatform::log::source logger;
    string host;
    unsigned short port = 443;
    nghttp2_session* nghttp2_native = nullptr;
    write_buffer_t<> write_buffer;
    std::array<uint8_t, 8 * 1024> read_buffer;
    boost::asio::mutable_buffers_1 asio_read_buffer;
    time_traits::time_point inited_at;

    // Session is idle when write is not active
    // and there are no pending requests.
    bool idle = false;

    std::deque<shared_request_data> pending_requests;
    std::map<int32_t, shared_request_data> active_requests;

    bool shutdown = false;

private:
    template <typename K, typename V>
    const V& get_value(const std::pair<K, V>& pair)
    {
        return pair.second;
    }

    template <typename T>
    const T& get_value(const T& t)
    {
        return t;
    }
};

using shared_session = shared_ptr<session>;

class session_pool
{
public:
    void attach(const shared_session& session)
    {
        sessions.push_back(session);
    }
    void detach(const shared_session& session)
    {
        sessions.erase(std::remove(sessions.begin(), sessions.end(), session));
    }
    size_t size() const
    {
        return sessions.size();
    }
    const shared_session& operator[](size_t n) const
    {
        return sessions[n];
    }

private:
    std::vector<shared_session> sessions;
};

using shared_session_pool = shared_ptr<session_pool>;

}}
