#pragma once

#include "session.h"
#include "utils.h"

#include <boost/asio/ssl/error.hpp>

namespace ymod_httpclient { namespace h2 {

static bool tls_h2_negotiated(session& session)
{
    auto ssl = session.socket.stream().get_ssl_stream()->native_handle();
    const unsigned char* next_proto = nullptr;
    unsigned int next_proto_len = 0;
    SSL_get0_alpn_selected(ssl, &next_proto, &next_proto_len);
    if (!next_proto)
    {
        YLOG(session.logger, info) << "protocol negotiation failed: no proto";
        return false;
    }
    if (!std::equal(next_proto, next_proto + next_proto_len, H2_PROTO))
    {
        YLOG(session.logger, info) << "protocol negotiation failed: unknown proto \""
                                   << utils::to_string(next_proto, next_proto_len) << "\"";
        return false;
    }
    return true;
}

static bool is_err_correct_eof(const boost::system::error_code& e)
{
    static auto short_read = boost::asio::ssl::error::stream_truncated;

    return e == boost::asio::error::eof || e == short_read || e == boost::asio::error::shut_down ||
        e == boost::asio::error::broken_pipe || e == boost::asio::error::connection_reset;
}

static bool want_io(const session& session, const error_code_t& ec)
{
    return !is_err_correct_eof(ec) || !session.idle ||
        nghttp2_session_want_read(session.nghttp2_native) ||
        nghttp2_session_want_write(session.nghttp2_native);
}

static void start_pending_timer(const shared_session& session, const shared_request_data& req)
{
    req->timer.expires_at(req->deadline);
    req->timer.async_wait([session, req_raw = req.get()](const boost::system::error_code& ec) {
        if (!ec)
        {
            session->fin_pending_request(req_raw, errc::request_timeout, "queue timeout");
        }
    });
}

static void start_active_timer(const shared_session& session, const shared_request_data& req)
{
    req->timer.expires_at(req->deadline);
    req->timer.async_wait(
        [session, stream_id = req->stream_id](const boost::system::error_code& ec) {
            if (!ec)
            {
                session->fin_active_request(stream_id, errc::request_timeout, "timeout");
            }
        });
}

}}
