#pragma once

#include "session.h"
#include "session_helpers.h"

#include <yplatform/time_traits.h>
#include <boost/asio.hpp>
#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>
#include <boost/noncopyable.hpp>

namespace ymod_httpclient { namespace h2 {

// Can be started by a client instance or connection control coroutine.
struct write_coroutine : public boost::asio::coroutine
{
    shared_session session;
    yplatform::log::source& logger;

    // For data produced by nghttp2 which doesn't fit into the write buffer.
    const uint8_t* overflow_data = nullptr;
    ssize_t overflow_data_len = 0;

    write_coroutine(const shared_session& session) : session(session), logger(session->logger)
    {
    }

    void operator()(const error_code_t& ec = error_code_t(), size_t /*bytes*/ = 0)
    {
        using namespace boost::asio;

        try
        {
            reenter(this)
            {
                assert(session->idle);
                session->idle = false;

                // TODO 1. configure send chunk size in settings;
                //      2. add stats for chunk sizes.
                for (;;)
                {
                    if (!prepare_write_buffer())
                    {
                        close(errc::protocol_error);
                        yield break;
                    }

                    while (submit_next_request())
                    {
                        if (!prepare_write_buffer())
                        {
                            close(errc::protocol_error);
                            yield break;
                        }
                        if (overflow_data) break;
                    }

                    if (session->write_buffer.empty())
                    {
                        yield break;
                    }

                    yield write();
                    if (ec == boost::asio::error::operation_aborted)
                    {
                        // Possible aborted by read timer, write timer
                        // or idle timeout (if no active requests).
                        close(errc::session_closed_error);
                        yield break;
                    }
                    else if (ec)
                    {
                        if (want_io(*session, ec))
                        {
                            YLOG(logger, error) << "write error: " << ec.message();
                        }
                        close(is_err_correct_eof(ec) ? errc::eof_error : errc::read_error);
                        yield break;
                    }
                    else
                    {
                        session->write_buffer.reset();
                        if (overflow_data)
                        {
                            pick_overflow_data();
                        }
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            try
            {
                YLOG(logger, error) << "write coro exception: " << e.what();
            }
            catch (...)
            {
            }
            try
            {
                session->socket.close();
            }
            catch (...)
            {
            }
        }

        if (is_complete())
        {
            session->idle = true;
        }
    }

    bool submit_next_request()
    {
        if (session->pending_requests.empty()) return false;
        auto req = session->pop_pending();
        auto stream_id = wrapped_nghttp2_submit_request(session, req);
        if (stream_id < 0)
        {
            // Possible errors: no mem, max id reached, invalid arg, proto error.
            // Connection will still be closed later.
            YLOG_CTX(session->logger, req->ctx, error)
                << "request submit error: " << nghttp2_strerror(stream_id);
            session->fin_request(req, errc::protocol_error);
        }
        else
        {
            req->stream_id = stream_id;
            if (!session->put_active(stream_id, req))
            {
                throw std::runtime_error("fatal error: duplicated stream ids");
            }
            else
            {
                start_active_timer(session, req);
            }
        }
        return true;
    }

    void pick_overflow_data()
    {
        session->write_buffer.pick(overflow_data, overflow_data_len);
        overflow_data = nullptr;
        overflow_data_len = 0;
    }

    bool prepare_write_buffer()
    {
        // nghttp2 emits data by small chunks so we need to aggregate
        // them until we run out buffer space.
        while (!overflow_data)
        {
            overflow_data_len = nghttp2_session_mem_send(session->nghttp2_native, &overflow_data);
            if (overflow_data_len == 0) break;
            if (overflow_data_len < 0)
            {
                YLOG(session->logger, error)
                    << "mem_send failed: " << nghttp2_strerror(overflow_data_len);
                return false;
            }
            if (overflow_data_len < session->write_buffer.free_space())
            {
                pick_overflow_data();
            }
        }
        return true;
    }

    void write()
    {
        session->socket.async_write(
            session->write_buffer.as_asio(), session->settings.write_timeout, *this);
    }

    void close(errc err)
    {
        session->socket.shutdown(true);
        session->fin_all_requests(err, "session closed");
    }
};

}}
