#pragma once

#include "header_parser.h"
#include <ymod_httpclient/types.h>
#include <yplatform/util.h>

namespace ymod_httpclient {

struct service_header_flags
{
    bool host = false;
    bool content_length = false;
    bool transfer_encoding = false;
    bool content_type = false;
    bool connection = false;
    bool x_request_id = false;
    bool x_request_timeout = false;
    bool x_request_attempt = false;
    bool x_ya_service_ticket = false;
};

class service_header_finder
{
public:
    service_header_finder(service_header_flags& flags) : flags_(flags)
    {
    }

    void operator()(const string& header_name)
    {
        if (yplatform::iequals(header_name, "Host"))
        {
            flags_.host = true;
        }
        else if (yplatform::iequals(header_name, "Content-Length"))
        {
            flags_.content_length = true;
        }
        else if (yplatform::iequals(header_name, "Content-Transfer-Encoding"))
        {
            flags_.transfer_encoding = true;
        }
        else if (yplatform::iequals(header_name, "Content-Type"))
        {
            flags_.content_type = true;
        }
        else if (yplatform::iequals(header_name, "Connection"))
        {
            flags_.connection = true;
        }
        else if (yplatform::iequals(header_name, "X-Request-Id"))
        {
            flags_.x_request_id = true;
        }
        else if (yplatform::iequals(header_name, "X-Request-Timeout"))
        {
            flags_.x_request_timeout = true;
        }
        else if (yplatform::iequals(header_name, "X-Request-Attempt"))
        {
            flags_.x_request_attempt = true;
        }
        else if (yplatform::iequals(header_name, "X-Ya-Service-Ticket"))
        {
            flags_.x_ya_service_ticket = true;
        }
    }

    void operator()(const string& name, [[maybe_unused]] const string& value)
    {
        operator()(name);
    }

private:
    service_header_flags& flags_;
};

inline string to_string(const header_dict& headers)
{
    // Chosen basing on use-cases in other services.
    const std::size_t EMPIRICAL_PREALLOC_SIZE = 128;

    string res;
    res.reserve(EMPIRICAL_PREALLOC_SIZE);
    for (const auto& header : headers)
    {
        res.append(header.first + ": " + header.second + "\r\n");
    }
    return res;
}

class header_container
{
public:
    header_container() = default;

    header_container(string headers)
    {
        update_flags_for_multiple_headers(headers);
        data_ = std::move(headers);
    }

    header_container(const header_dict& headers)
    {
        update_flags_for_multiple_headers(headers);
        data_ = to_string(headers);
    }

    void add(const string& name, const string& value)
    {
        update_flags_for_single_header(name);
        data_.append(name + ": " + value + "\r\n");
    }

    const string& data() const
    {
        return data_;
    }

    const service_header_flags& contained_headers() const
    {
        return contained_header_flags_;
    }

private:
    void update_flags_for_single_header(const string& header_name)
    {
        service_header_finder finder(contained_header_flags_);
        finder(header_name);
    }

    void update_flags_for_multiple_headers(const string& headers)
    {
        service_header_finder finder(contained_header_flags_);
        std::istringstream in(headers);
        parse_headers(in, finder);
    }

    void update_flags_for_multiple_headers(const header_dict& headers)
    {
        service_header_finder finder(contained_header_flags_);
        for (const auto& [name, value] : headers)
        {
            finder(name);
        }
    }

    string data_;
    service_header_flags contained_header_flags_;
};

inline std::ostream& operator<<(std::ostream& os, const header_container& headers)
{
    return os << headers.data();
}

}
