#pragma once

#include <yplatform/net/filter.h>
#include <utility>

namespace ymod_httpclient {

class http_filter : public yplatform::net::proto_filter
{
public:
    enum state
    {
        status_line,
        headers,
        content
    };

    http_filter() : session_state_(status_line), seen_nl_(false), seen_other_(false)
    {
    }

    void set_state(state session_state)
    {
        session_state_ = session_state;
        seen_nl_ = false; // just in case
        seen_other_ = false;
    }

    template <typename Iterator>
    std::pair<Iterator, bool> operator()(Iterator begin, Iterator end)
    {
        switch (session_state_)
        {
        case status_line:
            return filter_status_line(begin, end);
        case headers:
            return filter_headers(begin, end);
        default:
            return filter_content(begin, end);
        }
    }

private:
    template <typename Iterator>
    std::pair<Iterator, bool> filter_status_line(Iterator begin, Iterator end)
    {
        for (Iterator i = begin; i != end; ++i)
        {
            // Skip empty newlines before status line, possibly left
            // by a buggy server.
            seen_other_ = seen_other_ || (*i != '\r' && *i != '\n');
            if (seen_other_ && *i == '\n') return std::make_pair(i, true);
        }
        return std::make_pair(end, false);
    }

    template <typename Iterator>
    std::pair<Iterator, bool> filter_headers(Iterator begin, Iterator end)
    {
        for (Iterator i = begin; i != end; ++i)
        {
            if (*i == '\n')
            {
                if (seen_nl_) return std::make_pair(i, true);
                else
                    seen_nl_ = true;
            }
            else if (*i != '\r')
            {
                seen_nl_ = false;
            }
        }
        return std::make_pair(end, false);
    }

    template <typename Iterator>
    std::pair<Iterator, bool> filter_content(Iterator /*begin*/, Iterator end)
    {
        return std::make_pair(end - 1, true);
    }

    state session_state_;
    bool seen_nl_;
    bool seen_other_;
};

}

namespace boost { namespace asio {
template <>
struct is_match_condition<ymod_httpclient::http_filter> : public boost::true_type
{
};
}}
