#pragma once

#include <stdint.h>
#include <string.h>

namespace http_parser { namespace symbols {

#define F_CHAR 0x01 << 0
#define F_UPALPHA 0x01 << 1
#define F_LOALPHA 0x01 << 2
#define F_DIGIT 0x01 << 3
#define F_CTL 0x01 << 4
#define F_TSPECIALS 0x01 << 5
#define F_HEX 0x01 << 6

struct ascii_map
{
    ascii_map()
    {
        memset(http, F_CHAR, 256);
        for (int i = 'A'; i <= 'Z'; ++i)
            http[i] |= F_UPALPHA;
        for (int i = 'a'; i <= 'z'; ++i)
            http[i] |= F_LOALPHA;
        for (int i = '0'; i <= '9'; ++i)
            http[i] |= F_DIGIT;
        for (int i = 'A'; i <= 'F'; ++i)
            http[i] |= F_HEX;
        for (int i = 'a'; i <= 'f'; ++i)
            http[i] |= F_HEX;
        for (int i = '0'; i <= '9'; ++i)
            http[i] |= F_HEX;
        for (int i = 0; i < 32; ++i)
            http[i] |= F_CTL;
        http[127] |= F_CTL;
        http[static_cast<int>('(')] |= F_TSPECIALS;
        http[static_cast<int>(')')] |= F_TSPECIALS;
        http[static_cast<int>('<')] |= F_TSPECIALS;
        http[static_cast<int>('>')] |= F_TSPECIALS;
        http[static_cast<int>('@')] |= F_TSPECIALS;
        http[static_cast<int>(',')] |= F_TSPECIALS;
        http[static_cast<int>(';')] |= F_TSPECIALS;
        http[static_cast<int>(':')] |= F_TSPECIALS;
        http[static_cast<int>('\\')] |= F_TSPECIALS;
        http[static_cast<int>('"')] |= F_TSPECIALS;
        http[static_cast<int>('/')] |= F_TSPECIALS;
        http[static_cast<int>('[')] |= F_TSPECIALS;
        http[static_cast<int>(']')] |= F_TSPECIALS;
        http[static_cast<int>('?')] |= F_TSPECIALS;
        http[static_cast<int>('=')] |= F_TSPECIALS;
        http[static_cast<int>('{')] |= F_TSPECIALS;
        http[static_cast<int>('}')] |= F_TSPECIALS;
        http[static_cast<int>(' ')] |= F_TSPECIALS;
        http[static_cast<int>('\t')] |= F_TSPECIALS;
    }

    uint8_t http[256];
};

namespace {
const ascii_map symbols_map;
}

inline bool is_octet(uint8_t)
{
    return true;
}

inline bool is_char(uint8_t val)
{
    return symbols_map.http[val] & F_CHAR;
}

inline bool is_upalpha(uint8_t val)
{
    return symbols_map.http[val] & F_UPALPHA;
}

inline bool is_loalpha(uint8_t val)
{
    return symbols_map.http[val] & F_LOALPHA;
}

inline bool is_alpha(uint8_t val)
{
    return is_upalpha(val) || is_loalpha(val);
}

inline bool is_digit(uint8_t val)
{
    return symbols_map.http[val] & F_DIGIT;
}

inline bool is_ctl(uint8_t val)
{
    return symbols_map.http[val] & F_CTL;
}

inline bool is_tspecials(uint8_t val)
{
    return symbols_map.http[val] & F_TSPECIALS;
}

inline bool is_cr(uint8_t val)
{
    return val == '\r';
}

inline bool is_lf(uint8_t val)
{
    return val == '\n';
}

inline bool is_crlf(uint8_t val)
{
    return is_cr(val) || is_lf(val);
}

inline bool is_sp(uint8_t val)
{
    return val == ' ';
}

inline bool is_lws(uint8_t val)
{
    return is_crlf(val) || is_sp(val);
}

inline bool is_ht(uint8_t val)
{
    return val == '\t';
}

inline bool is_whitespace(uint8_t val)
{
    return is_sp(val) || is_ht(val);
}

inline bool is_dquote(uint8_t val)
{
    return val == '"';
}

inline bool is_text(uint8_t val)
{
    return (is_ctl(val) ? is_sp(val) || is_lf(val) || is_cr(val) || is_ht(val) : true);
}

inline bool is_hex(uint8_t val)
{
    return symbols_map.http[val] & F_HEX;
}

inline bool is_token(uint8_t val)
{
    return is_char(val) && !is_ctl(val) && !is_tspecials(val);
}

inline bool is_ctext(uint8_t val)
{
    return is_text(val) && val != '(' && val != ')';
}

inline bool is_qdtext(uint8_t val)
{
    return is_text(val) && val != '"';
}

}}
