#pragma once

#include <http_parser/char_sets.h>
#include <http_parser/common.h>
#include <ymod_httpclient/types.h>
#include <boost/algorithm/string/case_conv.hpp>
#include <string>
#include <map>

namespace http_parser {

typedef std::map<string, string> header_map_t;

template <typename Iterator>
class header_parser
{
    typedef Iterator (
        header_parser::*state_handler_t)(Iterator begin, Iterator& start, Iterator end);

public:
    header_parser() : current_handler_(&header_parser::on_header_begin), finished_(false)
    {
        i_header = headers.end();
    }

    Iterator operator()(Iterator begin, Iterator& start, Iterator end)
    {
        state_handler_t prev_handler;
        while (true)
        {
            prev_handler = current_handler_;
            begin = (this->*(current_handler_))(begin, start, end);
            if (prev_handler == current_handler_ || finished_ || begin == end) break;
        }
        return begin;
    }

    header_map_t result_headers()
    {
        return headers;
    } // @todo: move it!

    bool is_finished() const
    {
        return finished_;
    }

private:
    Iterator on_header_begin(Iterator begin, Iterator& start, Iterator end);
    Iterator on_header_name(Iterator begin, Iterator& start, Iterator end);
    Iterator on_header_value(Iterator begin, Iterator& start, Iterator end);
    Iterator on_header_skip_sp(Iterator begin, Iterator& start, Iterator end);
    Iterator on_header_skip_crlf(Iterator begin, Iterator& start, Iterator end);
    Iterator on_header_end_cr(Iterator begin, Iterator& start, Iterator end);

    state_handler_t current_handler_;
    header_map_t headers;
    header_map_t::iterator i_header;
    bool finished_;
};

template <typename Iterator>
Iterator header_parser<Iterator>::on_header_begin(Iterator begin, Iterator& start, Iterator end)
{
    if (start == end)
    {
        return begin;
    }

    if (symbols::is_crlf(*start))
    {
        current_handler_ = &header_parser::on_header_end_cr;
        return begin;
    }
    if (i_header != headers.end() && symbols::is_whitespace(*start))
    {
        current_handler_ = &header_parser::on_header_skip_sp;
        return ++start;
    }
    current_handler_ = &header_parser::on_header_name;
    return start;
}

template <typename Iterator>
Iterator header_parser<Iterator>::on_header_name(Iterator begin, Iterator& start, Iterator end)
{
    while (start != end && *start != ':')
        ++start;
    if (start == end) return begin;
    string name(boost::to_lower_copy(encode_param_range(begin, start)));
    i_header = headers.find(name);
    if (i_header == headers.end()) i_header = headers.insert(std::make_pair(name, "")).first;
    else if (i_header->second.size())
        i_header->second.append(", ");
    current_handler_ = &header_parser::on_header_skip_sp;
    return ++start;
}

template <typename Iterator>
Iterator header_parser<Iterator>::on_header_skip_sp(Iterator, Iterator& start, Iterator end)
{
    while (start != end && symbols::is_whitespace(*start))
        ++start;
    if (start != end) current_handler_ = &header_parser::on_header_value;
    return start;
}

template <typename Iterator>
Iterator header_parser<Iterator>::on_header_value(Iterator begin, Iterator& start, Iterator end)
{
    while (start != end && !symbols::is_crlf(*start))
        ++start;
    if (start == end) return begin;
    current_handler_ = &header_parser::on_header_skip_crlf;
    i_header->second.append(encode_header_range(begin, start));
    return start;
}

template <typename Iterator>
Iterator header_parser<Iterator>::on_header_skip_crlf(Iterator, Iterator& start, Iterator end)
{
    while (start != end && symbols::is_cr(*start))
        ++start;
    if (start == end) return start;
    if (!symbols::is_lf(*start))
        throw std::runtime_error("header_parser: invalid header finish sequence");
    current_handler_ = &header_parser::on_header_begin;
    return ++start;
}

template <typename Iterator>
Iterator header_parser<Iterator>::on_header_end_cr(Iterator, Iterator& start, Iterator end)
{
    while (start != end && symbols::is_cr(*start))
        ++start;
    if (start == end) return start;
    if (!symbols::is_lf(*start)) throw std::runtime_error("header_parser: invalid headers ending");
    finished_ = true;
    return ++start;
}

}
