#pragma once

#include "request.h"
#include <boost/format.hpp>
#include <optional>

namespace ymod_httpclient {

const string MPOST_BOUNDARY = "--------yplatform-http-client-data-boundary";
const string MPOST_BEGIN = "\r\n--" + MPOST_BOUNDARY + "\r\n";
const string MPOST_CONTENT_DISPOSITION_HEADER =
    "Content-Disposition: form-data; name=\"%1%\"\r\n\r\n";
const string MPOST_CONTENT_TYPE_HEADER = "Content-Type: %1%\r\n\r\n";
const string MPOST_END = "\r\n--" + MPOST_BOUNDARY + "--\r\n";
const string MPOST_DEFAULT_CONTENT_TYPE = "multipart/form-data";

struct status_line
{
    string http_version;
    unsigned int status_code = 0;
    string reason;
};

class end_of_data_characteristic
{
    enum state
    {
        CLenght,
        Chunked,
        Eof
    };

public:
    end_of_data_characteristic();

    void set_content_length_mode(size_t len);
    void set_chunked_mode();
    void set_wait_eof();

    bool is_chunked_mode();

    // Apply data and check should we stop (false) or read further (true)
    bool apply(size_t length);
    // return result of last apply()
    bool is_end();
    bool is_wait_eof();

private:
    state state_;
    size_t content_length_;
    bool got_last_chunk_;
};

inline const auto& get_part_header(const string& type)
{
    if (type.empty() || yplatform::iequals(type, MPOST_DEFAULT_CONTENT_TYPE))
    {
        return MPOST_CONTENT_DISPOSITION_HEADER;
    }
    return MPOST_CONTENT_TYPE_HEADER;
}

inline auto write_to_buffer(request_data_ptr req)
{
    auto host = req->remote_point;
    auto result = make_shared<boost::asio::streambuf>();
    std::ostream request_stream(result.get());
    switch (req->method)
    {
    case request::method_t::GET:
        request_stream << "GET";
        break;
    case request::method_t::POST:
        request_stream << "POST";
        break;
    case request::method_t::PUT:
        request_stream << "PUT";
        break;
    case request::method_t::HEAD:
        request_stream << "HEAD";
        break;
    case request::method_t::DELETE:
        request_stream << "DELETE";
        break;
    }
    request_stream << " " << host->uri_prefix + req->uri << " HTTP/1.1\r\n";

    if (!req->headers.contained_headers().host)
    {
        request_stream << "Host: " << host->host << "\r\n";
    }
    request_stream << req->headers;
    if (!req->headers.contained_headers().connection && !req->reuse_connection)
        request_stream << "Connection: close"
                       << "\r\n";

    if (req->method == request::method_t::POST || req->method == request::method_t::PUT)
    {
        if (req->post)
        {
            if (!req->headers.contained_headers().content_length)
            {
                request_stream << "Content-Length: " << req->post->size() << "\r\n";
            }
            if (!req->headers.contained_headers().transfer_encoding)
            {
                request_stream << "Content-Transfer-Encoding: 8bit\r\n";
            }
            if (!req->headers.contained_headers().content_type)
            {
                request_stream
                    << "Content-Type: application/x-www-form-urlencoded; charset=UTF-8\r\n";
            }
        }
        else if (req->mpost)
        {
            std::size_t length = 0;
            for (auto& [first, second] : req->mpost->parts)
            {
                if (second.empty()) continue;
                length += MPOST_BEGIN.length();
                length += get_part_header(req->mpost->type).length() - 3 + first.length();
                length += second.length();
            }
            if (length > 0)
            {
                length += MPOST_END.length();
            }
            if (!req->headers.contained_headers().content_type)
            {
                auto& content_type =
                    req->mpost->type.size() ? req->mpost->type : MPOST_DEFAULT_CONTENT_TYPE;
                request_stream << "Content-Type: " << content_type
                               << "; boundary=" << MPOST_BOUNDARY << "\r\n";
            }
            if (!req->headers.contained_headers().content_length)
            {
                request_stream << "Content-Length: " << length << "\r\n";
            }
        }
        else
        {
            request_stream << "Content-Length: 0\r\n";
        }
    }
    request_stream << "Accept: */*\r\n";
    request_stream << "\r\n";

    if (req->post)
    {
        request_stream << *req->post;
        return result;
    }

    if (!req->mpost) return result;

    bool written = false;
    for (auto& [first, second] : req->mpost->parts)
    {
        if (second.empty()) continue;
        request_stream << MPOST_BEGIN << boost::format(get_part_header(req->mpost->type)) % first
                       << second;
        written = true;
    }
    if (written)
    {
        request_stream << MPOST_END;
    }

    return result;
}

inline std::optional<status_line> parse_status_line(boost::asio::streambuf& buffer)
{
    status_line ret;
    std::istream response_stream(&buffer);
    response_stream >> ret.http_version;
    response_stream >> ret.status_code;
    string status_message;

    std::getline(response_stream, status_message);

    string::const_iterator status_begin = status_message.begin();

    while (status_begin != status_message.end() && *status_begin == ' ')
        ++status_begin;

    string::const_iterator status_end = status_begin;
    while (status_end != status_message.end() && *status_end != '\r' && *status_end != '\n')
        ++status_end;

    ret.reason = string(status_begin, status_end);
    return response_stream ? ret : std::optional<status_line>();
}

}
