#include <ymod_httpclient/errors.h>
#include <yplatform/time_traits.h>
#include <boost/container/flat_map.hpp>
#include <iterator>
#include <algorithm>
#include <vector>
#include <string>
#include <cstdint>

namespace ymod_httpclient {

constexpr bool is_registered_status_code(int code)
{
    return 100 <= code && code < 600;
}

struct request_processing_stats
{
    void increment(int status_code, unsigned attempt_no, http_error::code errc)
    {
        if (errc == http_error::code::success)
        {
            if (is_registered_status_code(status_code)) status_code_counters_[status_code]++;
            else
                non_registered_status_code_counter_++;
        }

        if (attempt_no < std::size(attempt_no_counters_)) attempt_no_counters_[attempt_no]++;
        else
            overflowing_attempt_no_counter_++;

        if (errc < static_cast<int>(std::size(error_counters_))) error_counters_[errc]++;
        else
            other_error_codes_counter_++;
    }

    void fill(yplatform::ptree& parent) const
    {
        parent.put_child("status_code_counts", status_code_stats());
        parent.put_child("attempt_no_counts", attempt_no_stats());
        parent.put_child("error_counts", error_stats());
    }

private:
    yplatform::ptree status_code_stats() const
    {
        yplatform::ptree result;
        for (auto&& [code, counter] : status_code_counters_)
        {
            if (counter)
                result.put(std::to_string(code) + "_cumulative", counter);
        }
        if (non_registered_status_code_counter_)
            result.put("non_registered_cumulative", non_registered_status_code_counter_);
        return result;
    }

    yplatform::ptree attempt_no_stats() const
    {
        yplatform::ptree result;
        for (std::size_t i = 0; i < std::size(attempt_no_counters_); i++)
        {
            if (request_count_t counter = attempt_no_counters_[i])
                result.put(std::to_string(i) + "_cumulative", counter);
        }
        if (overflowing_attempt_no_counter_)
            result.put("too_big_cumulative", overflowing_attempt_no_counter_);
        return result;
    }

    yplatform::ptree error_stats() const
    {
        yplatform::ptree result;
        for (unsigned errc = 0; errc < std::size(error_counters_); errc++)
        {
            if (request_count_t counter = error_counters_[errc])
            {
                std::string name = message(http_error::code{ static_cast<int>(errc) });
                std::replace(name.begin(), name.end(), ' ', '_');
                result.put(name + "_cumulative", counter);
            }
        }
        if (other_error_codes_counter_)
            result.put("other_errors_cumulative", other_error_codes_counter_);
        return result;
    }

    using request_count_t = std::uint_fast64_t;

    // An emperical value --- about twice as much as the max value usually configured.
    static constexpr std::size_t MAX_RECORDED_ATTEMPT_NO = 9;

    boost::container::flat_map<int, request_count_t> status_code_counters_;
    request_count_t non_registered_status_code_counter_ = 0;

    // Element #N is the count of N-th attempts.
    request_count_t attempt_no_counters_[1 + MAX_RECORDED_ATTEMPT_NO] = {};
    request_count_t overflowing_attempt_no_counter_ = 0;

    request_count_t error_counters_[http_error::code::COUNT] = {};
    request_count_t other_error_codes_counter_ = 0;
};

} // namespace ymod_httpclient
