#include "session.h"
#include "header_parser.h"
#include <yplatform/util.h>
#include <yplatform/log.h>

#include <boost/algorithm/string/case_conv.hpp>
#include <boost/asio/ssl/error.hpp>
#include <boost/bind.hpp>
#include <boost/range.hpp>
#include <boost/lexical_cast.hpp>

#include <cassert>
#include <sstream>

namespace asio = boost::asio;

namespace ymod_httpclient {

void replace_handler(
    request_data_ptr req,
    const std::function<void(http_error::code, const string&)>& handler)
{
    req->handler = [req_weak = weak_ptr<request_data>(req),
                    handler = handler,
                    original_handler = req->handler](auto&& err, auto&& reason) mutable {
        if (auto req = req_weak.lock(); req)
        {
            std::swap(req->handler, original_handler); // Restore original handler back.
        }
        handler(err, reason);
        handler = {};
        original_handler = {};
    };
}

using namespace http_parser;

template <class C>
class header_handler
{
public:
    header_handler(C& container, response_handler_ptr handler)
        : container_(container), handler_(handler)
    {
    }

    void operator()(const string& name, const string& value)
    {
        container_.emplace(std::make_pair(name, value));
        if (handler_)
        {
            handler_->handle_header(name, value);
        }
        if (name == "transfer-encoding")
        {
            transfer_encoding_ = value;
            boost::to_lower(transfer_encoding_);
        }
        else if (name == "connection")
        {
            connection_ = value;
            boost::to_lower(connection_);
        }
    }

    const string& transfer_encoding() const
    {
        return transfer_encoding_;
    }

    const string& connection() const
    {
        return connection_;
    }

private:
    C& container_;
    response_handler_ptr handler_;
    string transfer_encoding_;
    string connection_;
};

namespace p = std::placeholders;

session::session(
    yplatform::net::io_data& io_data,
    const remote_point_info& server,
    const yplatform::log::source& logger,
    const ymod_httpclient::settings& settings)
    : yplatform::log::contains_logger(logger)
    , settings_(settings)
    , socket_(io_data, settings_.socket)
    , read_buffer_(new buffer(std::numeric_limits<std::size_t>::max()))
    , use_ssl_(server.proto == "https")
    , reuse_connection_(false)
    , server_(server)
    , request_number_(0)
{
    if (server_.port == 0) server_.port = use_ssl_ ? 443 : 80;
}

void session::connect(
    yplatform::task_context_ptr ctx,
    const time_traits::duration& connect_timeout,
    const handler_type& handler)
{
    socket_.get_io()->post([ctx, connect_timeout, handler, this, self = this->shared_from_this()] {
        auto operation = make_shared<connect_op>(socket_, settings_.socket.resolve_order);
        operation->logger(logger());
        operation->set_log_prefix(logger().get_log_prefix() + ctx->uniq_id());
        auto connect_cb = std::bind(
            &session::handle_connect,
            this->shared_from_this(),
            p::_1,
            operation,
            time_traits::clock::now(),
            connect_timeout != time_traits::duration::max() ?
                time_traits::clock::now() + connect_timeout :
                time_traits::time_point::max(),
            handler);
        operation->perform(
            server_.host,
            server_.port,
            settings_.connect_attempt_timeout,
            connect_timeout,
            connect_cb);
    });
}

void session::run(request_data_ptr req, const handler_type& handler)
{
    socket_.get_io()->post([req, handler, this, self = this->shared_from_this()] {
        if (!is_reusable() && stats_.requests_processed > 0)
        {
            return handler(http_error::session_closed_error, "session expired");
        }
        replace_handler(req, handler);
        run_pending_request(req);
    });
}

void session::idle(const handler_type& handler)
{
    socket_.async_read(
        *read_buffer_,
        settings_.poll_timeout,
        [handler, self = this->shared_from_this()](
            const boost::system::error_code& err, std::size_t) mutable {
            if (err == boost::asio::error::operation_aborted)
            {
                handler({}, "");
            }
            else
            {
                handler(http_error::session_closed_error, "connection was closed by server");
            }
        });
}

void session::stop_idle()
{
    socket_.cancel_operations();
}

void session::shutdown()
{
    socket_.shutdown(true);
    // Start this operation to check that other side correct close session
    socket_.async_read(
        *read_buffer_,
        time_traits::duration::max(),
        [this, self = this->shared_from_this()](auto err, std::size_t) {
            if (!is_err_correct_eof(err))
            {
                YLOG_L(warning) << "server side did not close connection";
            }
        });
}

void session::close()
{
    reuse_connection_ = false;
    if (connected())
    {
        auto ptr = this->shared_from_this();
        socket_.cancel_operations();
        socket_.close();
    }
}

bool session::is_reusable() const
{
    return reuse_connection_ && connected();
}

void session::handle_connect(
    const boost::system::error_code& err,
    connect_op_ptr connect_op,
    const time_traits::time_point& start_time,
    const time_traits::time_point& deadline,
    const handler_type& handler)
{
    stats_.resolve_time = connect_op->total_resolve_time();
    stats_.connect_time = time_traits::clock::now() - start_time;
    if (err)
    {
        return handler(
            err == boost::asio::error::operation_aborted ? http_error::code::connection_timeout :
                                                           http_error::code::connect_error,
            err.message());
    }
    if (use_ssl_)
    {
        auto self = shared_from_this();
        auto handle_ssl_cb =
            std::bind(&session::handle_ssl, self, p::_1, time_traits::clock::now(), handler);
        if (settings_.send_ssl_server_name)
        {
            socket_.set_tls_server_name(server_.host.c_str());
        }
        if (settings_.ssl_verify_hostname)
        {
            socket_.set_tls_verify_callback(boost::asio::ssl::rfc2818_verification(server_.host));
        }
        socket_.async_tls_handshake(
            socket::handshake_type::client, deadline, std::move(handle_ssl_cb));
    }
    else
    {
        handler({}, "");
    }
}

void session::handle_ssl(
    const boost::system::error_code& err,
    const time_traits::time_point& start_time,
    const handler_type& handler)
{
    stats_.tls_time = time_traits::clock::now() - start_time;
    if (err)
    {
        return handler(
            err == boost::asio::error::operation_aborted ? http_error::code::request_timeout :
                                                           http_error::code::ssl_error,
            err.message());
    }
    handler({}, "");
}

bool session::connected() const
{
    return socket_.is_open();
}

void session::run_pending_request(request_data_ptr req)
{
    reuse_connection_ = false;
    stats_.requests_processed++;
    send_request(req);
}

void session::send_request(request_data_ptr req)
{
    auto req_processing_state = make_shared<request_processing_state>();
    auto buffer = write_to_buffer(req);
    socket_.async_write(
        *buffer,
        req->deadline(),
        [this, self = yplatform::shared_from(this), buffer, req_processing_state, req](
            auto err, size_t bytes) {
            handle_write_request(err, bytes, req_processing_state, req);
        });
}

void session::handle_write_request(
    const boost::system::error_code& err,
    std::size_t const& bytes,
    request_processing_state_ptr req_processing_state,
    request_data_ptr req)
{
    req->bytes_out += bytes;

    if (err)
    {
        if (err == boost::asio::error::operation_aborted)
        {
            return req->handler(http_error::code::request_timeout, "request write timeout");
        }
        else
        {
            return req->handler(
                is_err_correct_eof(err) ? http_error::code::eof_error :
                                          http_error::code::write_error,
                "request write error: " + err.message());
        }
    }
    read_response_status_line(req_processing_state, req);
}

void session::read_response_status_line(
    request_processing_state_ptr req_processing_state,
    request_data_ptr req)
{
    req_processing_state->filter.set_state(http_filter::status_line);
    socket_.async_read_until(
        *read_buffer_,
        req_processing_state->filter,
        req->deadline(),
        std::bind(
            &session::handle_read_status_line,
            this->shared_from_this(),
            p::_1,
            p::_2,
            req_processing_state,
            req));
}

void session::handle_read_status_line(
    const boost::system::error_code& err,
    std::size_t const& bytes,
    request_processing_state_ptr req_processing_state,
    request_data_ptr req)
{
    req->bytes_in += bytes;

    if (err)
    {
        if (err == boost::asio::error::operation_aborted)
        {
            return req->handler(http_error::code::request_timeout, "status read line timeout");
        }
        else
        {
            return req->handler(
                is_err_correct_eof(err) ? http_error::code::eof_error :
                                          http_error::code::read_error,
                "status read line error: " + err.message());
        }
    }

    // Check that response is OK.
    auto status_line = parse_status_line(*read_buffer_);
    if (!status_line || status_line->http_version.substr(0, 5) != "HTTP/")
    {
        req->handler(
            http_error::code::server_response_error,
            "bad HTTP version: " + (status_line ? status_line->http_version : ""));
        return;
    }
    req_processing_state->http_version = status_line->http_version;

    if (req->response_handler && req->response_handler->version() > handler_version_initial)
    {
        req->response_handler->set_code(status_line->status_code, status_line->reason);
    }
    else
    {
        if (status_line->status_code != 200)
        {
            req->handler(
                http_error::code::server_status_error,
                "server response code: " + std::to_string(status_line->status_code));
            return;
        }
    }
    read_response_headers(req_processing_state, req);
}

void session::read_response_headers(
    request_processing_state_ptr req_processing_state,
    request_data_ptr req)
{
    req_processing_state->filter.set_state(http_filter::headers);
    socket_.async_read_until(
        *read_buffer_,
        req_processing_state->filter,
        req->deadline(),
        std::bind(
            &session::handle_read_headers,
            this->shared_from_this(),
            p::_1,
            p::_2,
            req_processing_state,
            req));
}

void session::handle_read_headers(
    const boost::system::error_code& err,
    std::size_t const& bytes,
    request_processing_state_ptr req_processing_state,
    request_data_ptr req)
{
    req->bytes_in += bytes;

    if (err)
    {
        if (err == boost::asio::error::operation_aborted)
        {
            return req->handler(http_error::code::request_timeout, "headers read timeout");
        }
        else
        {
            return req->handler(
                http_error::code::read_error, "headers read error: " + err.message());
        }
    }

    // Process the response headers.
    std::istream response_stream(read_buffer_.get());
    header_handler<header_map> hdr_handler(req_processing_state->headers, req->response_handler);
    parse_headers(response_stream, hdr_handler);
    req_processing_state->chunk_mode = (hdr_handler.transfer_encoding() == "chunked");

    if (!specify_end_of_data_characteristic(req_processing_state, req)) return;

    // Write whatever content we already have to output.
    if (read_buffer_->size() > 0)
    {
        if (!process_content_data(req_processing_state, req)) return;
    }
    read_content(hdr_handler.connection(), req_processing_state, req);
}

bool session::specify_end_of_data_characteristic(
    request_processing_state_ptr req_processing_state,
    request_data_ptr req)
{
    header_map::const_iterator it;

    if (req->method == request::method_t::HEAD)
    {
        req_processing_state->is_end_of_data.set_content_length_mode(0);
        return true;
    }

    // Check: Content-Length
    it = req_processing_state->headers.find("content-length");
    if (it != req_processing_state->headers.end())
    {
        try
        {
            size_t length = boost::lexical_cast<int>(it->second);
            req_processing_state->is_end_of_data.set_content_length_mode(length);
            return true;
        }
        catch (const boost::bad_lexical_cast&)
        {
            req->handler(http_error::code::server_header_error, "incorrect Content-Length");
            return false;
        }
    }

    // Check: transfer-encoding = chunked
    it = req_processing_state->headers.find("transfer-encoding");
    if (it != req_processing_state->headers.end() && it->second == "chunked")
    {
        req_processing_state->is_end_of_data.set_chunked_mode();
        return true;
    }

    // Default way (not good)
    req_processing_state->is_end_of_data.set_wait_eof();
    YLOG_L(debug) << "read response until EOF: " << server_.proto << "://" << server_.host << ":"
                  << server_.port << server_.uri_prefix << req->uri;
    return true;
}

void session::read_content(
    const string& connection_header,
    request_processing_state_ptr req_processing_state,
    request_data_ptr req)
{
    if (!req_processing_state->is_end_of_data.is_end())
    {
        socket_.async_read(
            *read_buffer_,
            req->deadline(),
            std::bind(
                &session::handle_read_content,
                this->shared_from_this(),
                p::_1,
                p::_2,
                connection_header,
                req_processing_state,
                req));
    }
    else
    {
        update_reuse_connection(connection_header, req_processing_state, req);
        complete_request(req);
    }
}

void session::update_reuse_connection(
    const string& connection_header,
    request_processing_state_ptr req_processing_state,
    request_data_ptr req)
{
    reuse_connection_ = req->reuse_connection && req_processing_state->http_version == "HTTP/1.1" &&
        (connection_header.empty() || connection_header == "keep-alive");
}

void session::handle_read_content(
    const boost::system::error_code& err,
    std::size_t bytes,
    const string& connection_header,
    request_processing_state_ptr req_processing_state,
    request_data_ptr req)
{
    req->bytes_in += bytes;

    if (bytes)
    {
        // Write all of the data that has been read so far.
        // If process_content_data fail it would call the handler appropriate error.
        if (!process_content_data(req_processing_state, req)) return;
        if (!err)
        {
            read_content(connection_header, req_processing_state, req);
            return;
        }
    }
    if (err)
    {
        if ((req_processing_state->is_end_of_data.is_wait_eof() ||
             req_processing_state->is_end_of_data.is_end()) &&
            is_err_correct_eof(err))
        {
            reuse_connection_ = false;
            complete_request(req);
        }
        else
        {
            if (err == boost::asio::error::operation_aborted)
                req->handler(http_error::code::request_timeout, "content read timeout");
            else
                req->handler(http_error::code::read_error, "content read error: " + err.message());
        }
    }
}

bool session::process_content_data(
    request_processing_state_ptr req_processing_state,
    request_data_ptr req)
{
    if (req_processing_state->chunk_mode)
    {
        string consumed_data;
        auto start = boost::asio::buffers_begin(read_buffer_->data());
        auto end = boost::asio::buffers_end(read_buffer_->data());
        auto saved = start + req_processing_state->chunked_read_bytes;
        try
        {
            auto iconsume = req_processing_state->chunked_parser(start, saved, end, consumed_data);
            if (consumed_data.size())
            {
                if (!handle_content_data(consumed_data.data(), consumed_data.size(), req))
                {
                    return false;
                }
            }
            if (req_processing_state->chunked_parser.is_finished())
            {
                req_processing_state->is_end_of_data.apply(0);
            }
            std::size_t consume_size = iconsume - start;
            req_processing_state->chunked_read_bytes = saved - iconsume;
            read_buffer_->consume(consume_size);
        }
        catch (const std::exception& e)
        {
            req->handler(
                http_error::code::parse_response_error, string("chunk parser: ") + e.what());
            return false;
        }
    }
    else
    {
        for (const asio::const_buffer& buf : read_buffer_->data())
        {
            if (!handle_content_data(
                    asio::buffer_cast<const char*>(buf), asio::buffer_size(buf), req))
            {
                return false;
            }
            try
            {
                req_processing_state->is_end_of_data.apply(asio::buffer_size(buf));
            }
            catch (const std::exception& e)
            {
                req->handler(http_error::code::parse_response_error, e.what());
                return false;
            }
        }
        read_buffer_->consume(read_buffer_->size());
    }
    return true;
}

bool session::handle_content_data(const char* buff, size_t size, request_data_ptr req)
{
    try
    {
        if (req->response_handler) req->response_handler->handle_data(buff, size);
    }
    catch (const std::exception& e)
    {
        std::stringstream stream;
        yplatform::util::log_strip_cmd(stream, boost::make_iterator_range(buff, buff + size), 1024);
        req->handler(
            http_error::code::response_handler_error,
            string("handle content data error: ") + e.what());
        return false;
    }
    return true;
}

void session::complete_request(request_data_ptr req)
{
    try
    {
        handle_content_data_end(req);
    }
    catch (const std::exception& e)
    {
        req->handler(http_error::code::response_handler_error, e.what());
        return;
    }
}

void session::handle_content_data_end(request_data_ptr req)
{
    if (req->response_handler) req->response_handler->handle_data_end();
    req->handler({}, "");
}

bool session::is_err_correct_eof(const boost::system::error_code& err)
{
    // 'short read' error happens when work over ssl
    // it's not clear case so we just skip it
    static auto short_read = boost::asio::ssl::error::stream_truncated;

    return (
        (err == asio::error::eof) || (err == short_read) || (err == asio::error::shut_down) ||
        (err == asio::error::connection_reset) || (err == asio::error::connection_aborted));
}

}
