#pragma once

#include "protocol.h"
#include "http_filter.h"
#include "request.h"
#include <ymod_httpclient/settings.h>
#include <ymod_httpclient/types.h>
#include <ymod_httpclient/response_handler.h>
#include <http_parser/chunked.h>
#include <yplatform/net/socket.h>
#include <yplatform/net/sequental_connect.h>
#include <yplatform/time_traits.h>
#include <boost/function.hpp>
#include <boost/asio/buffers_iterator.hpp>

namespace ymod_httpclient {

struct service_header_flags;

class session
    : public enable_shared_from_this<session>
    , public yplatform::log::contains_logger
{
    typedef std::unordered_multimap<string, string> header_map;
    typedef boost::asio::streambuf buffer;
    typedef shared_ptr<buffer> buffer_ptr;
    typedef boost::asio::buffers_iterator<buffer::const_buffers_type> read_iterator;
    typedef std::function<void(http_error::code, const string&)> handler_type;
    typedef yplatform::net::tcp_socket socket;
    typedef yplatform::net::async_sequental_connect_op<socket> connect_op;
    typedef shared_ptr<connect_op> connect_op_ptr;

    struct request_processing_state
    {
        http_filter filter;
        bool chunk_mode = false;
        end_of_data_characteristic is_end_of_data;
        http_parser::chunked<read_iterator, string> chunked_parser;
        std::size_t chunked_read_bytes = 0;
        header_map headers;
        string http_version;
    };
    typedef shared_ptr<request_processing_state> request_processing_state_ptr;

public:
    struct stats
    {
        std::size_t requests_processed = 0;
        time_traits::duration resolve_time = {};
        time_traits::duration connect_time = {};
        time_traits::duration tls_time = {};
    };

    session(
        yplatform::net::io_data& io_data,
        const remote_point_info& server,
        const yplatform::log::source& logger,
        const ymod_httpclient::settings& settings);

    void connect(
        yplatform::task_context_ptr ctx,
        const time_traits::duration& connect_timeout,
        const handler_type& handler);

    void run(request_data_ptr req, const handler_type& handler);

    void idle(const handler_type& handler);

    void stop_idle();

    void shutdown();

    void close();

    bool is_reusable() const;

    void const* id() const
    {
        return socket_.id();
    }

    std::uint64_t get_next_request_number()
    {
        return ++request_number_;
    }

    const stats& stats()
    {
        return stats_;
    }

private:
    void handle_connect(
        const boost::system::error_code& err,
        connect_op_ptr connect_op,
        const time_traits::time_point& start_time,
        const time_traits::time_point& deadline,
        const handler_type& handler);

    void handle_ssl(
        const boost::system::error_code& err,
        const time_traits::time_point& start_time,
        const handler_type& handler);

    bool connected() const;
    void run_pending_request(request_data_ptr req);

    void send_request(request_data_ptr req);

    void handle_write_request(
        const boost::system::error_code& err,
        const std::size_t& bytes,
        request_processing_state_ptr req_processing_state,
        request_data_ptr req);

    void read_response_status_line(
        request_processing_state_ptr req_processing_state,
        request_data_ptr req);

    void handle_read_status_line(
        const boost::system::error_code& err,
        const std::size_t& bytes,
        request_processing_state_ptr req_processing_state,
        request_data_ptr req);

    void read_response_headers(
        request_processing_state_ptr req_processing_state,
        request_data_ptr req);

    void handle_read_headers(
        const boost::system::error_code& err,
        const std::size_t& bytes,
        request_processing_state_ptr req_processing_state,
        request_data_ptr req);

    // Until when we read the answer:
    //  * Content-Length
    //  * last chunk (transfer-encoding = chunked)
    //  * EOF (if nothing else specified)
    bool specify_end_of_data_characteristic(
        request_processing_state_ptr req_processing_state,
        request_data_ptr req);

    void read_content(
        const string& connection_header,
        request_processing_state_ptr req_processing_state,
        request_data_ptr req);

    void update_reuse_connection(
        const string& connection_header,
        request_processing_state_ptr req_processing_state,
        request_data_ptr req);

    void handle_read_content(
        const boost::system::error_code& err,
        std::size_t bytes,
        const string& connection_header,
        request_processing_state_ptr req_processing_state,
        request_data_ptr req);

    bool process_content_data(
        request_processing_state_ptr req_processing_state,
        request_data_ptr req);

    bool handle_content_data(const char* buf, size_t size, request_data_ptr req);

    void complete_request(request_data_ptr req);

    void handle_content_data_end(request_data_ptr req);

    bool is_err_correct_eof(const boost::system::error_code& err);

    const ymod_httpclient::settings& settings_;
    socket socket_;
    buffer_ptr read_buffer_;
    bool use_ssl_;
    std::atomic_bool reuse_connection_;
    remote_point_info server_;
    std::uint64_t request_number_;
    struct stats stats_;
};

}
