#pragma once

#include "request_processing_stats.h"
#include <ymod_httpclient/types.h>
#include <ymod_httpclient/errors.h>
#include <yplatform/task_context.h>
#include <yplatform/reactor.h>
#include <yplatform/reactor/tracer.h>
#include <atomic>
#include <deque>
#include <mutex>

namespace ymod_httpclient {

namespace ph = std::placeholders;

template <class Session, typename CongestionController>
class session_pool : public enable_shared_from_this<session_pool<Session, CongestionController>>
{
    typedef Session session;
    typedef CongestionController congestion_controller;
    typedef session_pool<Session, congestion_controller> this_type;
    typedef weak_ptr<session> weak_session_ptr;
    typedef shared_ptr<session> session_ptr;
    typedef std::deque<weak_session_ptr> sessions;
    typedef std::unique_lock<std::mutex> lock;
    typedef std::function<session_ptr(boost::asio::io_service&)> factory_method;
    typedef std::function<void(http_error::code, const string&, session_ptr)> session_cb;

public:
    session_pool(
        yplatform::reactor_ptr reactor,
        std::size_t preferred_sessions_count,
        const time_traits::duration& reactor_overload_delay)
        : reactor_(reactor)
        , reactor_overload_delay_(reactor_overload_delay)
        , preferred_sessions_count_(preferred_sessions_count)
        , acquired_sessions_(0)
    {
        connect_controllers_.reserve(reactor->size());
        for (size_t i = 0; i < reactor->size(); ++i)
        {
            connect_controllers_[(*reactor)[i]->io()] = congestion_controller();
        }
    }

    ~session_pool()
    {
        close();
    }

    session_ptr get()
    {
        session_ptr result;
        {
            lock lock(mux_);
            while (sessions_.size())
            {
                auto session = sessions_.front().lock();
                sessions_.pop_front();
                if (session && session->is_reusable())
                {
                    result = session;
                    break;
                };
            }
        }
        if (result)
        {
            result->stop_idle();
            lock lock(mux_);
            acquired_sessions_++;
        }
        return result;
    }

    void create(
        yplatform::task_context_ptr ctx,
        const factory_method& factory_method,
        const time_traits::duration& connect_timeout,
        const session_cb& cb)
    {
        auto io = reactor_->io();
        if (!controller_try_run(io))
        {
            return cb(http_error::task_throttled, "connect throttled", nullptr);
        }
        auto session = factory_method(*io);
        session->connect(
            ctx,
            connect_timeout,
            std::bind(
                &this_type::handle_connect,
                yplatform::shared_from(this),
                ph::_1,
                ph::_2,
                io,
                session,
                cb));
        lock lock(mux_);
        acquired_sessions_++;
    }

    void put(const session_ptr& sess)
    {
        if (!sess)
        {
            return;
        }
        lock lock(mux_);
        if (acquired_sessions_ <= 0)
        {
            throw std::runtime_error("session_pool error: return not acquired session");
        }
        --acquired_sessions_;
        if (!sess->is_reusable()) return;
        if (sessions_.size() >= preferred_sessions_count_)
        {
            sess->shutdown();
        }
        else
        {
            sess->idle([sess](auto&& err, auto&&) {
                if (err)
                {
                    sess->close();
                }
            });
            sessions_.emplace_back(sess);
        }
    }

    void close()
    {
        lock lock(mux_);
        sessions sessions;
        sessions.swap(sessions_);
        lock.unlock();
        for (auto& weak_session : sessions)
        {
            if (auto session = weak_session.lock()) session->close();
        }
    }

    size_t size() const
    {
        lock lock(mux_);
        return sessions_.size();
    }

    size_t acquired_size() const
    {
        lock lock(mux_);
        return acquired_sessions_;
    }

    void increment_request_processing_stats(
        int status_code,
        unsigned attempt_no,
        http_error::code errc)
    {
        lock lock(mux_);
        request_processing_stats_.increment(status_code, attempt_no, errc);
    }

    yplatform::ptree stats() const
    {
        yplatform::ptree ret;
        lock lock(mux_);
        ret.put("active-connections", acquired_sessions_);
        ret.put("idle-connections", sessions_.size());
        ret.put_child("congestion-control", controller_stats());
        request_processing_stats_.fill(ret);
        return ret;
    }

private:
    bool controller_try_run(boost::asio::io_service* io)
    {
        auto tracer = yplatform::find_tracer(*io);
        auto controller = find_controller(io);
        if (!tracer || !controller)
        {
            return true;
        }
        lock lock(mux_);
        return controller->try_run();
    }

    void handle_connect(
        http_error::code err,
        const string& reason,
        boost::asio::io_service* io,
        session_ptr session,
        const session_cb& cb)
    {
        controller_finish(io);
        cb(err, reason, session);
    }

    void controller_finish(boost::asio::io_service* io)
    {
        auto tracer = yplatform::find_tracer(*io);
        auto controller = find_controller(io);
        if (!tracer || !controller)
        {
            return;
        }
        lock lock(mux_);
        if (tracer->last_delay() > reactor_overload_delay_)
        {
            controller->finish_with_congestion();
        }
        else
        {
            controller->finish_without_congestion();
        }
    }

    congestion_controller* find_controller(boost::asio::io_service* io)
    {
        auto it = connect_controllers_.find(io);
        return it != connect_controllers_.end() ? &it->second : nullptr;
    }

    yplatform::ptree controller_stats() const
    {
        yplatform::ptree controller_stats;
        for (size_t i = 0; i < reactor_->size(); ++i)
        {
            auto it = connect_controllers_.find((*reactor_)[i]->io());
            if (it == connect_controllers_.end()) continue;
            controller_stats.push_back(
                std::make_pair(reactor_->name() + std::to_string(i), it->second.stats()));
        }
        return controller_stats;
    }

    mutable std::mutex mux_;
    yplatform::reactor_ptr reactor_;
    time_traits::duration reactor_overload_delay_;
    sessions sessions_;
    std::size_t preferred_sessions_count_;
    std::size_t acquired_sessions_;
    std::unordered_map<boost::asio::io_service*, congestion_controller> connect_controllers_;
    request_processing_stats request_processing_stats_;
};

}
