#ifndef _YMOD_IMAPCLIENT_IMAP_RESPONSE_H_
#define _YMOD_IMAPCLIENT_IMAP_RESPONSE_H_

#include <ymod_imapclient/config.h>
#include <yplatform/time_traits.h>
#include <vector>

namespace ymod_imap_client {

//-----------------------------------------------------------------------------
// Utf7/Utf8 folder names

struct Utf7ImapMailboxName;

struct Utf8MailboxName : protected std::string
{
    char delim = 0;

    Utf8MailboxName();
    Utf8MailboxName(const std::string& name, char delim);
    Utf8MailboxName(const Utf7ImapMailboxName& utf7Name);

    bool isParentOf(const Utf8MailboxName&) const;
    char getDelimiter()
    {
        return delim;
    }
    const std::string& asString() const
    {
        return *this;
    }
};

struct Utf7ImapMailboxName : protected std::string
{
    char delim;

    Utf7ImapMailboxName(const Utf8MailboxName& utf8Name);
    Utf7ImapMailboxName(const std::string& name, char delim);

    const std::string& asString() const
    {
        return *this;
    }
};

//-----------------------------------------------------------------------------
// Utf7/Utf8 labels

struct Utf7ImapLabel : protected std::string
{
    static Utf7ImapLabel create(const std::string& label)
    {
        return Utf7ImapLabel(label);
    }
    static Utf7ImapLabel create(std::string&& label)
    {
        return Utf7ImapLabel(std::move(label));
    }

    const std::string& asString() const
    {
        return *this;
    }

protected:
    Utf7ImapLabel(const std::string& label) : std::string(label)
    {
    }
    Utf7ImapLabel(std::string&& label) : std::string(std::move(label))
    {
    }
};

struct Utf8Label : public std::string
{
    Utf8Label() = default;
    Utf8Label(const Utf7ImapLabel& utf7Label);

    const std::string& asString() const
    {
        return *this;
    }
};

//-----------------------------------------------------------------------------
// Basic response types

struct ListResponse
{
    std::string name;
    uint32_t flags = 0;
    char delim = 0;

    enum FolderFlags : uint32_t
    {
        ff_none = 0,
        ff_noinferiors = 1,
        ff_noselect = 1 << 1,
        ff_marked = 1 << 2,
        ff_unmarked = 1 << 3,
        ff_hasnochildren = 1 << 4,

        // xlist & IMAP LIST Extension for Special-Use Mailboxes
        ff_drafts = 1 << 5,
        ff_sent = 1 << 6,
        ff_trash = 1 << 7,
        ff_spam = 1 << 8,
        ff_all = 1 << 9,
        ff_important = 1 << 10,
        ff_starred = 1 << 11,
        ff_inbox = 1 << 12
    };

    bool operator==(const ListResponse& that) const
    {
        return name == that.name && flags == that.flags && delim == that.delim;
    }
};

struct MailboxInfoResponse
{
    MailboxInfoResponse() = default;
    MailboxInfoResponse(
        uint32_t exists,
        uint32_t recent,
        uint32_t uidnext,
        uint32_t uidvalidity,
        uint32_t unseen)
        : exists(exists), recent(recent), uidnext(uidnext), uidvalidity(uidvalidity), unseen(unseen)
    {
    }

    uint32_t exists = 0;
    uint32_t recent = 0;
    uint32_t uidnext = 0;
    uint32_t uidvalidity = 0;
    uint32_t unseen = 0;

    bool operator==(const MailboxInfoResponse& that) const
    {
        return exists == that.exists && recent == that.recent && uidnext == that.uidnext &&
            uidvalidity == that.uidvalidity && unseen == that.unseen;
    }
};

struct FetchResponse
{
    uint32_t num = 0;
    uint32_t uid = 0;
    uint32_t flags = 0;
    uint64_t size = 0;
    std::vector<Utf8Label> xgmlabels;
    std::tm internaldate = { 0, 0, 0, 1, 0, 70, 0, 0, 0, 0, nullptr };

    enum MessageFlags : uint32_t
    {
        mf_none = 0,
        mf_answered = 1 << 0,
        mf_flagged = 1 << 1,
        mf_deleted = 1 << 2,
        mf_seen = 1 << 3,
        mf_draft = 1 << 4,
        mf_recent = 1 << 5,
    };

    std::shared_ptr<std::string> body;

    bool operator==(const FetchResponse& that) const
    {
        auto& t = internaldate;
        auto& that_t = that.internaldate;
        auto this_tie =
            std::tie(t.tm_sec, t.tm_min, t.tm_hour, t.tm_gmtoff, t.tm_year, t.tm_mon, t.tm_mday);
        auto that_tie = std::tie(
            that_t.tm_sec,
            that_t.tm_min,
            that_t.tm_hour,
            that_t.tm_gmtoff,
            that_t.tm_year,
            that_t.tm_mon,
            that_t.tm_mday);

        return num == that.num && uid == that.uid && flags == that.flags &&
            xgmlabels == that.xgmlabels && body == that.body && this_tie == that_tie;
    }

    std::time_t internaldate_to_time() const
    {
        auto date_copy = internaldate; // time conversion functions modifies source struct
        auto gmotff = date_copy.tm_gmtoff;
        date_copy.tm_gmtoff = 0;

        auto res = timegm(&date_copy);
        res -= gmotff;
        return res;
    }
};

struct FetchBodyPrefixResponse
{
    uint32_t size;
};

} // namespace ymod_imap_client

#endif // _YMOD_IMAPCLIENT_IMAP_RESPONSE_H_
