#ifndef _YMOD_IMAPCLIENT_IMAP_RESULT_H_
#define _YMOD_IMAPCLIENT_IMAP_RESULT_H_

#include <yplatform/future/future.hpp>

#include <ymod_imapclient/config.h>
#include <ymod_imapclient/imapuidls.h>
#include <ymod_imapclient/imap_response.h>

#include <set>
#include <deque>

namespace ymod_imap_client {

typedef boost::shared_ptr<std::string> StringPtr;
typedef yplatform::future::future<StringPtr> FutureString;

//-----------------------------------------------------------------------------
// Connect result

struct ConnectResult
{
    explicit ConnectResult(const std::string& serverIp = "") : serverIp(serverIp)
    {
    }

    std::string serverIp;
};

typedef std::shared_ptr<ConnectResult> ConnectResultPtr;
typedef yplatform::future::future<ConnectResultPtr> FutureConnectResult;

//-----------------------------------------------------------------------------
// Base IMAP command result

struct ImapResult
{
    ImapResult() = default;
    explicit ImapResult(bool ok) : ok(ok)
    {
    }
    virtual ~ImapResult()
    {
    }

    bool ok = true;
};

typedef std::shared_ptr<ImapResult> ImapResultPtr;
typedef yplatform::future::future<ImapResultPtr> FutureImapResult;

//-----------------------------------------------------------------------------
// Imap capabilities

struct Capability
{
    bool id = false;
    bool idle = false;
    bool move = false;
    bool xlist = false;
    bool xoauth2 = false;
    bool gmailEx = false;
    bool starttls = false;
    bool authPlain = false;

    std::set<string> items;
};

struct ImapCapability : public ImapResult
{
    ImapCapability(Capability&& capability);

    Capability capability;
};

typedef std::shared_ptr<ImapCapability> ImapCapabilityPtr;
typedef yplatform::future::future<ImapCapabilityPtr> FutureCapability;

//-----------------------------------------------------------------------------
// Result of LIST/LSUB/XLIST command

struct ImapListItem;
typedef std::shared_ptr<ImapListItem> ImapListItemPtr;
typedef std::vector<ImapListItemPtr> ImapMailboxList;

struct ImapListItem
{
    ImapListItem() = delete;
    ImapListItem(const Utf8MailboxName& name, uint32_t flags);

    Utf8MailboxName name;
    uint32_t flags = 0;
};

struct ImapList : public ImapResult
{
    ImapMailboxList mailboxes;

    ImapList(ImapMailboxList&& mailboxes);
};

typedef std::shared_ptr<ImapList> ImapListPtr;
typedef yplatform::future::future<ImapListPtr> FutureImapList;

//-----------------------------------------------------------------------------
// Result of EXAMINE/SELECT command

struct ImapMailboxResult : public ImapResult
{
    ImapMailboxResult(MailboxInfo&& mailboxInfo);

    MailboxInfo mailboxInfo;
};

typedef std::shared_ptr<ImapMailboxResult> ImapMailboxResultPtr;
typedef yplatform::future::future<ImapMailboxResultPtr> FutureImapMailbox;

struct ImapStatus : public ImapResult
{
    ImapStatus(MailboxInfo&& mailboxInfo);

    MailboxInfo mailboxInfo;
};

typedef std::shared_ptr<ImapStatus> ImapStatusPtr;
typedef yplatform::future::future<ImapStatusPtr> FutureImapStatus;

//-----------------------------------------------------------------------------
// Message UID set

struct ImapMessage
{
    uint32_t num = 0;
    uint32_t uid = 0;
    uint32_t flags = 0;
    uint64_t size = 0;

    std::time_t internaldate = 0;
    std::vector<Utf8Label> gmailLabels;
    std::string body;
};

typedef std::deque<ImapMessage> ImapMessageVector;
bool operator==(const ImapMessageVector& lhs, const ImapMessageVector& rhs);

struct MessageSet : public ImapResult
{
    MessageSet(ImapMessageVector&& messages);

    ImapMessageVector messages;
};

typedef std::shared_ptr<MessageSet> MessageSetPtr;
typedef yplatform::future::future<MessageSetPtr> FutureMessageSet;

//-----------------------------------------------------------------------------
// ID result

struct IdResult : public ImapResult
{
    IdResult(std::map<string, string>&& data);

    std::map<string, string> data;
};

typedef std::shared_ptr<IdResult> IdResultPtr;
typedef yplatform::future::future<IdResultPtr> FutureIdResult;

struct CopyuidResult : public ImapResult
{
    CopyuidResult() = default;
    CopyuidResult(
        uint32_t uidvalidity,
        std::vector<uint32_t>&& oldUids,
        std::vector<uint32_t>&& newUids);

    uint32_t uidvalidity = 0;

    std::vector<uint32_t> oldUids;
    std::vector<uint32_t> newUids;
};

typedef std::shared_ptr<CopyuidResult> CopyuidResultPtr;
typedef yplatform::future::future<CopyuidResultPtr> FutureCopyuidResult;

struct AppenduidResult : public ImapResult
{
    AppenduidResult() = default;
    AppenduidResult(uint32_t uidvalidity, uint32_t uid);

    uint32_t uidvalidity = 0;
    uint32_t uid = 0;
};

typedef std::shared_ptr<AppenduidResult> AppenduidResultPtr;
typedef yplatform::future::future<AppenduidResultPtr> FutureAppenduidResult;

} // namespace ymod_imap_client

#endif // _YMOD_IMAPCLIENT_IMAP_RESULT_H_
