#include "client_session.h"
#include "commands/command_connect.hpp"
#include "commands/command_base.hpp"
#include "commands/command_login.hpp"
#include "commands/command_authenticate.hpp"
#include "commands/command_list.hpp"
#include "commands/command_examine.hpp"
#include "commands/command_fetch.hpp"
#include "commands/command_mod.hpp"
#include "util/imap_utils.h"

#include <yplatform/encoding/base64.h>

#include <boost/algorithm/string/predicate.hpp>

namespace ymod_imap_client {

std::atomic_size_t debugCounter{ 0 };
std::atomic_size_t sessionCounter{ 0 };

size_t ClientSession::debugSessionCounter()
{
    return sessionCounter;
}

ClientSession::ClientSession(
    yplatform::net::base_service* service,
    const yplatform::net::client_settings& settings)
    : parent_t(service, settings), debugId(debugCounter++)
{
    sessionCounter++;
    LDEBUG_(this, get_context()) << "IMAP ClientSession#" << debugId
                                 << " created. Total: " << sessionCounter;
}

ClientSession::~ClientSession()
{
    sessionCounter--;
    LDEBUG_(this, get_context()) << "IMAP ClientSession#" << debugId
                                 << " destroyed. Total: " << sessionCounter;
}

void ClientSession::processCommandState(CommandState state)
{
    boost::apply_visitor(*this, state);
}

FutureConnectResult ClientSession::connect(
    const std::string& server,
    const string& serverLoggingAlias,
    unsigned port,
    bool sslSession)
{
    auto connectCommand =
        (sslSession ? std::make_shared<ConnectImapTLS>(nextTag()) :
                      std::make_shared<ConnectImap>(nextTag()));
    connectCommand->logger(logger());

    strand().post(
        [connectCommand, server, serverLoggingAlias, port, self = shared_from_this(), this]() {
            if (state != SessionState::Initial)
            {
                auto exception = connectCommand->makeExceptionWithServerResponse(
                    ConnectException("session not in initial state"));
                connectCommand->fulfillPromiseError(exception.e);
                return;
            }
            logger().append_log_prefix(serverLoggingAlias);
            this->server = serverLoggingAlias;
            state = SessionState::Busy;
            connectCommand->setContext(externalContext);
            connectCommand->setVerbositySettings(verbosity);
            runningCommand = connectCommand;
            commandCounter++;

            if (verbosity.serviceInfo)
            {
                LINFO_(this, get_context()) << "connecting to server:" << server;
            }
            auto connectCallback =
                boost::bind(&ClientSession::handleConnect, shared_from_this(), _1);
            auto timeoutCallback = boost::bind(&ClientSession::handleTimeout, shared_from_this());
            parent_t::connect(
                server, static_cast<unsigned short>(port), connectCallback, timeoutCallback);
        });

    return connectCommand->future();
}

void ClientSession::handleConnect(const boost::system::error_code& err)
{
    strand().post([err, self = shared_from_this(), this]() {
        if (!checkError<connect_error>(err)) return;
        auto commandState = runningCommand->handleConnection(remote_addr());
        processCommandState(commandState);
    });
}

void ClientSession::doStartTls()
{
    if (verbosity.serviceInfo)
    {
        LINFO_(this, get_context()) << "starting ssl";
    }

    auto tlsCallback = boost::bind(&ClientSession::handleTls, shared_from_this(), _1);
    auto timeoutCallback = boost::bind(&ClientSession::handleTimeout, shared_from_this());
    ClientSession::parent_t::start_tls(tlsCallback, timeoutCallback);
}

void ClientSession::handleTls(const boost::system::error_code& err)
{
    strand().post([err, self = shared_from_this(), this]() {
        if (!checkError<ssl_error>(err)) return;

        sslActive = true;
        auto commandState = runningCommand->handleTls();
        processCommandState(commandState);
    });
}

void ClientSession::readData(ImapFilterState filterState, std::size_t atleast)
{
    strand().post([filterState, atleast, self = shared_from_this(), this]() {
        if (verbosity.serviceInfo)
        {
            LINFO_(this, get_context()) << "recieving welcome message";
        }

        filter_.setState(filterState);

        auto dataCallback = boost::bind(&ClientSession::handleResponse, shared_from_this(), _1, _2);
        auto timeoutCallback = boost::bind(&ClientSession::handleTimeout, shared_from_this());

        if (!runningCommand)
        {
            return;
        }

        auto duration = runningCommand->commandDuration();
        if (duration >= commandTimeout)
        {
            return timeoutCallback();
        }

        auto timeout = commandTimeout - duration;
        if (atleast > 0) async_read(dataCallback, timeoutCallback, timeout, *buffer_, atleast);
        else
            async_read_until(dataCallback, timeoutCallback, timeout, *buffer_, filter_);
    });
}

void ClientSession::sendData(string&& commandText, ImapFilterState filterState)
{
    auto bufferPtr = boost::make_shared<StreamBuffer>();
    bufferPtr->sputn(commandText.data(), commandText.size());
    strand().post([bufferPtr, filterState, self = shared_from_this(), this]() {
        filter_.setState(filterState);

        auto timeoutCallback = boost::bind(&ClientSession::handleTimeout, shared_from_this());
        auto writeCallbak = [timeoutCallback, self, this](
                                const boost::system::error_code& err,
                                std::size_t /* size */,
                                StreamBufferPtr /* buff */) {
            strand().post([err, timeoutCallback, self, this]() {
                if (!checkError<transport_error>(err) || !runningCommand) return;

                auto duration = runningCommand->commandDuration();
                if (duration >= commandTimeout)
                {
                    return timeoutCallback();
                }

                auto timeout = commandTimeout - duration;
                auto responseCallback =
                    boost::bind(&ClientSession::handleResponse, shared_from_this(), _1, _2);
                async_read_until(responseCallback, timeoutCallback, timeout, *buffer_, filter_);
            });
        };

        if (!runningCommand)
        {
            return;
        }

        auto duration = runningCommand->commandDuration();
        if (duration >= commandTimeout)
        {
            return timeoutCallback();
        }

        auto timeout = commandTimeout - duration;
        sentBytes += bufferPtr->size();
        async_write_shared_buff(bufferPtr, writeCallbak, timeoutCallback, timeout);
    });
}

void ClientSession::handleResponse(const boost::system::error_code& err, std::size_t size)
{
    strand().post([err, size, self = shared_from_this(), this]() {
        if (runningCommand)
        {
            logCommandDuration(runningCommand->name(), runningCommand->commandDuration());
        }
        if (!checkError<transport_error>(err)) return;

        using Iterator = boost::asio::buffers_iterator<StreamBuffer::const_buffers_type>;
        auto begin = Iterator::begin(buffer_->data());
        auto response = std::make_shared<std::string>(begin, begin + size);

        receivedBytes += size;
        buffer_->consume(size);
        auto commandState = runningCommand->handleResponse(response);
        processCommandState(commandState);
    });
}

void ClientSession::handleTimeout()
{
    strand().post([self = shared_from_this(), this]() {
        if (runningCommand)
        {
            logCommandDuration(runningCommand->name(), runningCommand->commandDuration());
            processCommandState(runningCommand->makeExceptionWithServerResponse(
                TimeoutException(runningCommand->debugDump(), "Timeout expire.")));
        }
        try
        {
            cancel_operations();
        }
        catch (...)
        {
        }
        state = SessionState::Failure;
    });
}

FutureCapability ClientSession::capability()
{
    return exec<CommandCapability>();
}

FutureIdResult ClientSession::id(const string& idData)
{
    return exec<CommandId>(idData);
}

FutureImapResult ClientSession::startTls()
{
    return exec<CommandStartTls>();
}

FutureImapResult ClientSession::login(const std::string& user, const std::string& pass)
{
    return exec<CommandLogin>(user, pass);
}

FutureImapResult ClientSession::loginOauth(
    const std::string& user,
    const std::string& token,
    ImapClient::OauthLoginType type)
{
    std::string oauthData = "user=" + user + "\1auth=Bearer " + token + "\1\1";
    auto encoded = yplatform::base64_encode(oauthData.begin(), oauthData.end());
    return exec<CommandAuth>("XOAUTH2", std::string(encoded.begin(), encoded.end()), type);
}

FutureImapResult ClientSession::authPlain(const std::string& user, const std::string& pass)
{
    std::string authData;

    authData += '\0';
    authData += user;

    authData += '\0';
    authData += pass;

    auto encoded = yplatform::base64_encode(authData.begin(), authData.end());
    return exec<CommandAuth>("PLAIN", std::string(encoded.begin(), encoded.end()));
}

FutureImapList ClientSession::list(const std::string& refName, const std::string& mailbox)
{
    return exec<CommandList>(refName, mailbox);
}

FutureImapStatus ClientSession::status(const Utf8MailboxName& mailbox, const std::string& fields)
{
    return exec<CommandStatus>(mailbox, fields);
}

FutureImapMailbox ClientSession::examine(const Utf8MailboxName& mailbox)
{
    return exec<CommandExamine>(mailbox);
}

FutureImapResult ClientSession::select(const Utf8MailboxName& mailbox)
{
    return exec<CommandSelect>(mailbox);
}

FutureMessageSet ClientSession::fetch(const string& seqset, const string& args)
{
    return exec<CommandFetch>(seqset, args);
}

FutureMessageSet ClientSession::uidFetch(const string& seqset, const string& args)
{
    return exec<CommandUidFetch>(seqset, args);
}

FutureImapResult ClientSession::noop()
{
    return exec<CommandNoop>();
}

FutureImapResult ClientSession::logout()
{
    return exec<CommandLogout>();
}

FutureImapResult ClientSession::createMailbox(const Utf8MailboxName& mailbox)
{
    return exec<CommandCreate>(mailbox);
}

FutureImapResult ClientSession::deleteMailbox(const Utf8MailboxName& mailbox)
{
    return exec<CommandDelete>(mailbox);
}

FutureImapResult ClientSession::renameMailbox(
    const Utf8MailboxName& oldMailbox,
    const Utf8MailboxName& newMailbox)
{
    return exec<CommandRename>(oldMailbox, newMailbox);
}

FutureImapResult ClientSession::expunge()
{
    return exec<CommandExpunge>();
}

FutureMessageSet ClientSession::uidStore(
    const std::string& seqset,
    const std::string& storeType,
    const std::string& args)
{
    return exec<CommandUidStore>(seqset, storeType, args);
}

FutureCopyuidResult ClientSession::uidMove(
    const std::string& seqset,
    const Utf8MailboxName& newMailbox)
{
    return exec<CommandUidMove>(seqset, newMailbox);
}

FutureCopyuidResult ClientSession::uidCopy(
    const std::string& seqset,
    const Utf8MailboxName& newMailbox)
{
    return exec<CommandUidCopy>(seqset, newMailbox);
}

FutureAppenduidResult ClientSession::append(
    std::string&& body,
    const Utf8MailboxName& mailbox,
    const std::string& flags,
    const std::string& date)
{
    return exec<CommandAppend>(std::forward<std::string>(body), mailbox, flags, date);
}

void ClientSession::operator()(CommandFinished&)
{
    // Usually client sends new command from cb, so it must be called after state restoration
    auto cb = [cmd = runningCommand]() { cmd->fulfillPromise(); };
    runningCommand = nullptr;
    state = SessionState::Ready;
    strand().post(cb);
}

void ClientSession::operator()(CommandError& error)
{
    // Usually client sends new command from cb, so it must be called after state restoration
    auto cb = [cmd = runningCommand, error]() { cmd->fulfillPromiseError(error.e); };
    runningCommand = nullptr;
    state = SessionState::Ready;
    strand().post(cb);
}

} // namespace ymod_imap_client
