#ifndef _YMOD_IMAPCLIENT_CLIENT_SESSION_H_
#define _YMOD_IMAPCLIENT_CLIENT_SESSION_H_

#include "commands/command.hpp"
#include "util/imap_filter.hpp"

#include <ymod_imapclient/call.h>
#include <ymod_imapclient/errors.h>
#include <ymod_imapclient/statistics.h>

#include <yplatform/future/future.hpp>
#include <yplatform/net/client_session_strand.h>
#include <yplatform/time_traits.h>

#include <boost/enable_shared_from_this.hpp>
#include <boost/noncopyable.hpp>

#include <algorithm>
#include <sstream>
#include <iomanip>
#include <map>
#include <atomic>
#include <mutex>

namespace ymod_imap_client {

struct transport_error_greetings : public transport_error
{
};
template <typename ErrorType>
string getErrorPrefix();

template <>
inline string getErrorPrefix<connect_error>()
{
    return "connect to server error: ";
}
template <>
inline string getErrorPrefix<ssl_error>()
{
    return "error starting ssl: ";
}
template <>
inline string getErrorPrefix<transport_error_greetings>()
{
    return "error in handle_greetings: ";
}
template <>
inline string getErrorPrefix<transport_error>()
{
    return "error recieving data: ";
}

struct CmdTag
{
    int i_ = 0;

    std::string next()
    {
        i_++;
        std::ostringstream os;
        os << "A" << std::setfill('0') << std::setw(5) << i_;
        return os.str();
    }
};

enum class SessionState
{
    Initial,
    Ready,
    Failure,
    Busy
};

class ClientSession
    : public yplatform::net::client_session_strand<>
    , public boost::enable_shared_from_this<ClientSession>
    , public boost::static_visitor<>
    , private boost::noncopyable
{
    using StreamBuffer = boost::asio::basic_streambuf<std::allocator<char>>;
    using StreamBufferPtr = boost::shared_ptr<StreamBuffer>;

public:
    static size_t debugSessionCounter();

    typedef yplatform::net::client_session_strand<> parent_t;
    typedef boost::system::error_code error_code;

    ClientSession(
        yplatform::net::base_service* service,
        const yplatform::net::client_settings& settings);
    virtual ~ClientSession();

    virtual yplatform::task_context_ptr get_context() const
    {
        return externalContext;
    }
    void set_context(yplatform::task_context_ptr context)
    {
        if (context)
        {
            LDEBUG_(this, context)
                << "IMAP ClientSession#" << debugId << " context set:" << context->uniq_id();
        }
        externalContext = context;
    }
    void set_verbose(const VerbositySettings& verbosity)
    {
        this->verbosity = verbosity;
    }
    const VerbositySettings& get_verbose()
    {
        return this->verbosity;
    }

    FutureConnectResult connect(
        const std::string& server,
        const string& serverLoggingAlias,
        unsigned port,
        bool ssl);
    FutureCapability capability();
    FutureIdResult id(const string& idData);
    FutureImapResult startTls();
    FutureImapResult login(const std::string& user, const std::string& pass);
    FutureImapResult loginOauth(
        const std::string& user,
        const std::string& token,
        ImapClient::OauthLoginType type);
    FutureImapResult authPlain(const std::string& user, const std::string& pass);
    FutureImapList list(const std::string& refName = "", const std::string& mailbox = "*");
    FutureImapStatus status(const Utf8MailboxName& mailbox, const std::string& fields);
    FutureImapMailbox examine(const Utf8MailboxName& mailbox);
    FutureImapResult select(const Utf8MailboxName& mailbox);

    FutureMessageSet fetch(const string& seqset, const string& fetchArgs);
    FutureMessageSet uidFetch(const string& seqset, const string& fetchArgs);

    FutureImapResult noop();

    FutureImapResult logout();

    FutureImapResult createMailbox(const Utf8MailboxName& mailbox);
    FutureImapResult deleteMailbox(const Utf8MailboxName& mailbox);
    FutureImapResult renameMailbox(
        const Utf8MailboxName& oldMailbox,
        const Utf8MailboxName& newMailbox);

    FutureImapResult expunge();

    FutureMessageSet uidStore(
        const std::string& seqset,
        const std::string& storeType,
        const std::string& args);

    FutureCopyuidResult uidMove(const std::string& seqset, const Utf8MailboxName& newMailbox);
    FutureCopyuidResult uidCopy(const std::string& seqset, const Utf8MailboxName& newMailbox);
    FutureAppenduidResult append(
        std::string&& body,
        const Utf8MailboxName& mailbox,
        const std::string& flags,
        const std::string& date);

    bool isSecure() const
    {
        return sslActive;
    }

    Statistics getStats()
    {
        Statistics stats;
        stats.commandCount = commandCounter.load();
        stats.receivedBytes = receivedBytes.load();
        stats.sentBytes = sentBytes.load();
        return stats;
    }

    void setCommandTimeout(const yplatform::time_traits::duration& timeout)
    {
        if (timeout < yplatform::time_traits::duration::max())
        {
            commandTimeout = timeout;
        }
    }

    void operator()(Continue&)
    {
    }
    void operator()(StartTls&)
    {
        doStartTls();
    }
    void operator()(SendData& send)
    {
        sendData(std::move(send.data), send.filterState);
    }
    void operator()(ReadData& read)
    {
        readData(read.filterState, read.atleast);
    }
    void operator()(CommandFinished&);
    void operator()(CommandError& error);

private:
    string nextTag()
    {
        return cmdid_.next();
    }

    void doStartTls();
    void readData(ImapFilterState filterState, std::size_t atleast);
    void sendData(string&& commandText, ImapFilterState filterState);

    void processCommandState(CommandState state);

    template <typename ErrorType>
    bool checkError(const boost::system::error_code& err)
    {
        // Failure state means we alredy responded to client in timeout handler and should just exit
        // now
        if (state == SessionState::Failure) return false;
        if (state != SessionState::Busy)
        {
            LERR_(this, get_context())
                << getErrorPrefix<ErrorType>() << "wrong session state in checkError";
            if (runningCommand)
            {
                auto errorState = runningCommand->makeException(ErrorType() << BOOST_ERROR_INFO);
                processCommandState(errorState);
            }
            return false;
        }

        if (!err) return true;

        LERR_(this, get_context()) << getErrorPrefix<ErrorType>() << err.message();
        auto errorState = runningCommand->makeException(
            ErrorType() << yplatform::system_error(err) << BOOST_ERROR_INFO);
        processCommandState(errorState);
        return false;
    }

    void handleConnect(const boost::system::error_code& err);
    void handleTls(const boost::system::error_code& err);
    void handleResponse(const boost::system::error_code& err, std::size_t size);
    bool checkResponse(const grammar::ParseResult& parsed);
    void handleTimeout();

    template <class ImapCommandClass, class... Args>
    typename ImapCommandClass::FutureResultType exec(Args&&... args)
    {
        auto command = std::make_shared<ImapCommandClass>(nextTag(), std::forward<Args>(args)...);
        command->logger(logger());

        strand().post([command, self = shared_from_this(), this]() {
            if (state != SessionState::Ready)
            {
                auto exception = command->makeExceptionWithServerResponse(
                    ConnectException("session is not ready"));
                command->fulfillPromiseError(exception.e);
                return;
            }
            state = SessionState::Busy;
            command->setContext(externalContext);
            command->setVerbositySettings(verbosity);
            runningCommand = command;
            commandCounter++;

            if (verbosity.clientRequest)
            {
                LINFO_(this, get_context()) << "sending: " << command->debugDump();
            }
            sendData(std::move(command->makeCommand()), command->filterState());
        });

        return command->future();
    }

    void logCommandDuration(std::string commandName, uint32_t duration_ms)
    {
        std::replace(commandName.begin(), commandName.end(), ' ', '_');
        pa::async_profiler::add(
            pa::imap_client,
            externalContext ? externalContext->uniq_id() : "empty_context",
            server,
            commandName,
            duration_ms);
    }

    void logCommandDuration(
        std::string commandName,
        const yplatform::time_traits::duration& duration)
    {
        auto duration_ms =
            yplatform::time_traits::duration_cast<yplatform::time_traits::milliseconds>(duration);
        logCommandDuration(commandName, static_cast<uint32_t>(duration_ms.count()));
    }

    SessionState state = SessionState::Initial;
    VerbositySettings verbosity = VerbositySettings(false);

    StreamBufferPtr buffer_ = boost::make_shared<StreamBuffer>();
    ImapFilter filter_;
    yplatform::task_context_ptr externalContext;

    CmdTag cmdid_;

    ImapCommandPtr runningCommand;
    yplatform::time_traits::duration commandTimeout = yplatform::time_traits::milliseconds(5000);
    ;

    bool sslActive = false;
    size_t debugId = 0;
    std::atomic<uint32_t> commandCounter{ 0 };
    std::atomic<uint64_t> receivedBytes{ 0 };
    std::atomic<uint64_t> sentBytes{ 0 };
    std::string server;
};

typedef boost::shared_ptr<ClientSession> ClientSessionPtr;

} // namespace ymod_imap_client

#endif // _YMOD_IMAPCLIENT_CLIENT_SESSION_H_
