#ifndef _YMOD_IMAPCLIENT_COMMAND_H_
#define _YMOD_IMAPCLIENT_COMMAND_H_

#include "command_state.h"

#include <grammar/imap_base.hpp>
#include <util/imap_utils.h>
#include <util/imap_filter.hpp>

#include <ymod_imapclient/errors.h>
#include <ymod_imapclient/verbosity_settings.h>

#include <yplatform/future/future.hpp>
#include <yplatform/task_context.h>
#include <yplatform/time_traits.h>

#include <pa/async.h>

#include <boost/algorithm/string/regex.hpp>
#include <boost/regex.hpp>
#include <boost/algorithm/string.hpp>

namespace ymod_imap_client {

using namespace std::string_literals;

class ImapCommand
    : public std::enable_shared_from_this<ImapCommand>
    , public yplatform::log::contains_logger
{
public:
    virtual ~ImapCommand()
    {
    }

    virtual void setContext(yplatform::task_context_ptr ctx)
    {
        context = ctx;
    }
    virtual void setVerbositySettings(const VerbositySettings& newVerbosity)
    {
        verbosity = newVerbosity;
    }

    virtual ImapFilterState filterState() const
    {
        return ImapFilterState::EXPECT_RESPONSE_DONE;
    }
    virtual const string& tag() const = 0;
    virtual const string& name() const = 0;
    virtual const string& args() const = 0;
    virtual string makeCommand() const
    {
        return "";
    }
    virtual string debugDump() const = 0;
    virtual const VerbositySettings& getVerbose() const
    {
        return verbosity;
    }

    virtual CommandState handleConnection(const boost::asio::ip::address& /* remoteIp */)
    {
        return Continue();
    }
    virtual CommandState handleTls()
    {
        return Continue();
    }

    virtual CommandState handleResponse(std::shared_ptr<std::string> response)
    {
        lastResponse = response;
        auto parsed = parseResponse(response);
        if (getVerbose().serverResponse || !parsed)
        {
            logCommandResponse(response);
        }

        if (!parsed)
        {
            return CommandError{ makeExceptionWithServerResponse(
                FatalException(debugDump(), "cannot parse command response"s)) };
        }

        auto error = checkResponse(parsed->taggedResponse);
        if (error) return *error;

        return handleCheckedResponse(parsed);
    }

    virtual grammar::ParseResultPtr parseResponse(std::shared_ptr<std::string> response)
    {
        grammar::ParseResult res;
        bool parseOk = boost::spirit::qi::parse(
            response->begin(),
            response->end(),
            grammar::ResponseDone<std::string::iterator>(),
            res);
        if (!parseOk) return nullptr;
        return std::make_shared<grammar::ParseResult>(std::move(res));
    }

    virtual CommandState handleCheckedResponse(grammar::ParseResultPtr parsed) = 0;
    virtual void fulfillPromise() = 0;
    virtual void fulfillPromiseError(std::exception_ptr e) = 0;

    template <typename Exception>
    CommandError makeException(Exception&& e)
    {
        try
        {
            throw e;
        }
        catch (...)
        {
            auto ex = std::current_exception();
            // set_real_exception(ex);
            return CommandError{ ex };
        }
    }

    template <typename Exception>
    CommandError makeExceptionWithServerResponse(Exception&& e)
    {
        e.serverResponse = lastResponse;
        try
        {
            throw e;
        }
        catch (...)
        {
            auto ex = std::current_exception();
            return CommandError{ ex };
        }
    }

    yplatform::time_traits::duration commandDuration()
    {
        return yplatform::time_traits::clock::now() - commandStart;
    }

protected:
    boost::optional<CommandError> checkResponse(const grammar::TaggedResponse& parsed)
    {
        if (parsed.status == grammar::ResponseStatus::OK) return boost::optional<CommandError>();

        if (parsed.status == grammar::ResponseStatus::BAD)
        {
            return makeExceptionWithServerResponse(BadException(debugDump(), parsed.reason));
        }
        else if (parsed.status == grammar::ResponseStatus::NO)
        {
            return makeExceptionWithServerResponse(NoException(debugDump(), parsed.reason));
        }
        else
        {
            LERR_(this, context) << "Fatal error in onResponse for \"" << debugDump()
                                 << "\":" << parsed.reason;
            return makeExceptionWithServerResponse(FatalException(debugDump(), parsed.reason));
        }
    }

    void logCommandResponse(std::shared_ptr<std::string> response)
    {
        std::vector<std::string> lines;
        boost::algorithm::split_regex(lines, *response, boost::regex("\r\n"));
        lines.pop_back(); // last, empty line

        for (auto& line : lines)
        {
            LINFO_(this, context) << line;
        }
    }

protected:
    yplatform::task_context_ptr context;
    VerbositySettings verbosity = VerbositySettings(false);
    yplatform::time_traits::time_point commandStart = yplatform::time_traits::clock::now();
    std::shared_ptr<std::string> lastResponse;
};

typedef std::shared_ptr<ImapCommand> ImapCommandPtr;

template <typename ImapResultType>
std::shared_ptr<ImapResultType> makeParsedResult(grammar::ParseResultPtr parsed)
{
    return std::make_shared<ImapResultType>(
        parsed->taggedResponse.status == grammar::ResponseStatus::OK);
}

template <class ImapResultType>
class CommandBase : public ImapCommand
{
public:
    typedef std::shared_ptr<ImapResultType> ImapResultTypePtr;
    typedef yplatform::future::future<ImapResultTypePtr> FutureResultType;
    typedef yplatform::future::promise<ImapResultTypePtr> PromiseResultType;

    CommandBase(const string& tag, const string& name, const string& args = "")
        : commandTag(tag), commandName(name), commandArgs(args)
    {
    }

    virtual ~CommandBase()
    {
    }

    virtual const string& tag() const
    {
        return commandTag;
    };
    virtual const string& name() const
    {
        return commandName;
    };
    virtual const string& args() const
    {
        return commandArgs;
    }

    virtual string makeCommand() const
    {
        std::ostringstream request;
        request << tag() << " " << name();
        if (!args().empty()) request << " " << args();
        request << "\r\n";
        return request.str();
    }

    virtual string debugDump() const
    {
        return tag() + " " + name() + " " + args();
    }

    virtual CommandState handleCheckedResponse(grammar::ParseResultPtr parsed)
    {
        try
        {
            result = makeParsedResult<ImapResultType>(parsed);
            return CommandFinished();
        }
        catch (std::exception& e)
        {
            return CommandError{ std::current_exception() };
        }
    }

    virtual void fulfillPromise()
    {
        promise.set(result);
    }
    virtual void fulfillPromiseError(std::exception_ptr e)
    {
        promise.set_exception(e);
    }

    FutureResultType future()
    {
        return promise;
    }

protected:
    const std::string commandTag;
    const std::string commandName;
    const std::string commandArgs;
    std::shared_ptr<ImapResultType> result;
    PromiseResultType promise;
};

} // namespace ymod_imap_client

#endif // _YMOD_IMAPCLIENT_COMMAND_H_
