#ifndef _YMOD_IMAPCLIENT_COMMAND_AUTHENTICATE_H_
#define _YMOD_IMAPCLIENT_COMMAND_AUTHENTICATE_H_

#include "command.hpp"
#include <boost/lexical_cast.hpp>

namespace ymod_imap_client {

//-----------------------------------------------------------------------------
// Logging in

class CommandAuth : public CommandBase<ImapResult>
{
public:
    CommandAuth(
        const std::string& tag,
        const std::string& method,
        const std::string& data,
        ImapClient::OauthLoginType type = ImapClient::Multiline)
        : CommandBase(tag, "AUTHENTICATE", provideCommandArgs(method, data, type))
        , data(data + "\r\n")
        , authState(type == ImapClient::SinglelineQuoted ? AUTH_SENT : AUTH_INIT)
    {
    }

    virtual ImapFilterState filterState() const
    {
        return ImapFilterState::EXPECT_COMMAND_CONTINUATION;
    }

    virtual CommandState handleResponse(std::shared_ptr<std::string> response)
    {
        if ((*response)[0] == '+')
        {
            if (authState == AUTH_SENT)
            {
                return SendData{ "\r\n", ImapFilterState::EXPECT_COMMAND_CONTINUATION };
            }
            if (authState == AUTH_INIT)
            {
                authState = AUTH_SENT;
                return SendData{ std::move(data), ImapFilterState::EXPECT_COMMAND_CONTINUATION };
            }
            return Continue();
        }
        return CommandBase<ImapResult>::handleResponse(response);
    }

    virtual grammar::ParseResultPtr parseResponse(std::shared_ptr<std::string> response)
    {
        grammar::TaggedResponse parsed;
        bool parseOk = boost::spirit::qi::parse(
            response->begin(),
            response->end(),
            grammar::CommandContinuation<std::string::iterator>(),
            parsed);

        auto res = std::make_shared<grammar::ParseResult>();
        if (parseOk)
        {
            res->taggedResponse = parsed;
        }
        else
        {
            parseOk = boost::spirit::qi::parse(
                response->begin(),
                response->end(),
                grammar::ResponseDone<std::string::iterator>(),
                *res);
        }
        return parseOk ? res : nullptr;
    }

    virtual string debugDump() const
    {
        return tag() + " " + name() + " <authenticate data>";
    }

private:
    string data;

    enum AuthState
    {
        AUTH_INIT,
        AUTH_SENT
    };

    AuthState authState;

    static string provideCommandArgs(
        const std::string& method,
        const std::string& data,
        ImapClient::OauthLoginType type)
    {
        switch (type)
        {
        case ImapClient::Multiline:
            return method;

        case ImapClient::SinglelineQuoted:
            return method + " \"" + data + "\"";

        case ImapClient::SinglelineLiteral:
            return method + " {" + boost::lexical_cast<string>(data.size()) + "}";
        }

        return method;
    }
};

} // namespace ymod_imap_client

#endif // _YMOD_IMAPCLIENT_COMMAND_AUTHENTICATE_H_
