#ifndef _YMOD_IMAPCLIENT_COMMAND_BASE_H_
#define _YMOD_IMAPCLIENT_COMMAND_BASE_H_

#include "command.hpp"

namespace ymod_imap_client {

template <>
std::shared_ptr<ImapCapability> makeParsedResult(grammar::ParseResultPtr parsed)
{
    Capability capability;

    for (auto& item : parsed->capabilityData)
    {
        capability.items.insert(boost::algorithm::to_upper_copy(item));
    }

    auto end = capability.items.end();
    capability.id = capability.items.find("ID") != end;
    capability.idle = capability.items.find("IDLE") != end;
    capability.move = capability.items.find("MOVE") != end;
    capability.xlist = capability.items.find("XLIST") != end;
    capability.xoauth2 = capability.items.find("AUTH=XOAUTH2") != end;
    capability.gmailEx = capability.items.find("X-GM-EXT-1") != end;
    capability.starttls = capability.items.find("STARTTLS") != end;
    capability.authPlain = capability.items.find("AUTH=PLAIN") != end;

    return std::make_shared<ImapCapability>(std::move(capability));
}

template <>
std::shared_ptr<IdResult> makeParsedResult(grammar::ParseResultPtr parsed)
{
    std::map<string, string> items;
    for (size_t i = 0; i + 1 < parsed->idData.size(); i += 2)
    {
        items.insert(std::make_pair(parsed->idData[i], parsed->idData[i + 1]));
    }

    return std::make_shared<IdResult>(std::move(items));
}

//-----------------------------------------------------------------------------
// Basic commands like ID, CAPABILITY and so on

class CommandCapability : public CommandBase<ImapCapability>
{
public:
    CommandCapability(const std::string& tag) : CommandBase(tag, "CAPABILITY")
    {
    }
};

class CommandId : public CommandBase<IdResult>
{
public:
    CommandId(const std::string& tag, const std::string& clientIdData)
        : CommandBase(tag, "ID", "(" + clientIdData + ")")
    {
    }
};

class CommandStartTls : public CommandBase<ImapResult>
{
public:
    CommandStartTls(const std::string& tag) : CommandBase(tag, "STARTTLS")
    {
    }

    virtual CommandState handleCheckedResponse(grammar::ParseResultPtr parsed)
    {
        this->parsed = parsed;
        return StartTls();
    }

    virtual CommandState handleTls()
    {
        try
        {
            result = makeParsedResult<ImapResult>(parsed);
            return CommandFinished();
        }
        catch (std::exception& e)
        {
            return CommandError{ std::current_exception() };
        }
    }

protected:
    grammar::ParseResultPtr parsed;
};

class CommandLogout : public CommandBase<ImapResult>
{
public:
    CommandLogout(const std::string& tag) : CommandBase(tag, "LOGOUT")
    {
    }

    grammar::ParseResultPtr parseResponse(std::shared_ptr<std::string> response) override
    {
        grammar::ParseResult res;
        bool parseOk = boost::spirit::qi::parse(
            response->begin(), response->end(), grammar::ResponseBye<std::string::iterator>(), res);
        if (!parseOk) return nullptr;
        return std::make_shared<grammar::ParseResult>(std::move(res));
    }
};

class CommandNoop : public CommandBase<ImapResult>
{
public:
    CommandNoop(const std::string& tag) : CommandBase(tag, "NOOP")
    {
    }
};

} // namespace ymod_imap_client

#endif // _YMOD_IMAPCLIENT_COMMAND_BASE_H_
