#ifndef _YMOD_IMAPCLIENT_COMMAND_CONNECT_H_
#define _YMOD_IMAPCLIENT_COMMAND_CONNECT_H_

#include "command.hpp"
#include <grammar/imap_base.hpp>

namespace ymod_imap_client {

template <>
std::shared_ptr<ConnectResult> makeParsedResult(grammar::ParseResultPtr /*parsed*/)
{
    return std::make_shared<ConnectResult>();
}

//-----------------------------------------------------------------------------
// Connecting to remote server

class ConnectImap : public CommandBase<ConnectResult>
{
    typedef CommandBase<ConnectResult> ParentType;
    grammar::Greeting<std::string::iterator> greetingGrammar;

public:
    ConnectImap(const std::string& tag) : CommandBase(tag, "CONNECT")
    {
    }

    FutureConnectResult future()
    {
        return promise;
    }

    virtual CommandState handleConnection(const boost::asio::ip::address& remoteIp)
    {
        remoteAddress = remoteIp.to_string();
        return recieveGreetings();
    }

    virtual CommandState recieveGreetings()
    {
        return ReadData{ ImapFilterState::EXPECT_GREETING, 0 };
    }

    virtual grammar::ParseResultPtr parseResponse(std::shared_ptr<std::string> response)
    {
        grammar::TaggedResponse parsed;
        bool parseOk = boost::spirit::qi::parse(
            response->begin(), response->end(), grammar::Greeting<std::string::iterator>(), parsed);
        if (!parseOk) return nullptr;
        auto res = std::make_shared<grammar::ParseResult>();
        res->taggedResponse = parsed;
        return res;
    }

    virtual CommandState handleCheckedResponse(grammar::ParseResultPtr /*parsed*/)
    {
        result = std::make_shared<ConnectResult>(*remoteAddress);
        return CommandFinished();
    }

protected:
    boost::optional<std::string> remoteAddress;
};

class ConnectImapTLS : public ConnectImap
{
public:
    ConnectImapTLS(const std::string& tag) : ConnectImap(tag)
    {
    }

    virtual CommandState handleConnection(const boost::asio::ip::address& remoteIp)
    {
        remoteAddress = remoteIp.to_string();
        return StartTls();
    }

    virtual CommandState handleTls()
    {
        return recieveGreetings();
    }
};

} // namespace ymod_imap_client

#endif // _YMOD_IMAPCLIENT_COMMAND_CONNECT_H_
