#ifndef _YMOD_IMAPCLIENT_COMMAND_LIST_H_
#define _YMOD_IMAPCLIENT_COMMAND_LIST_H_

#include "command.hpp"
#include <grammar/list_grammar.hpp>
#include <ymod_imapclient/imap_result.h>

namespace ymod_imap_client {

template <>
inline std::shared_ptr<ImapList> makeParsedResult(grammar::ParseResultPtr parsed)
{
    static const std::string inboxName = "INBOX";

    ImapMailboxList mailboxes;

    for (auto& listItem : parsed->listResponses)
    {
        // Encoding convertation in constructor
        Utf8MailboxName utf8folderName(Utf7ImapMailboxName(listItem.name, listItem.delim));

        // INBOX is case insensitive special name, so we need to normalize it
        // Because of some strange responses like:
        //
        //* LIST (\HasChildren) "/" "INBOX"
        //* LIST (\HasNoChildren) "/" "Inbox/habr"
        if (boost::iequals(utf8folderName.asString(), inboxName))
        {
            utf8folderName = Utf8MailboxName(inboxName, listItem.delim);
        }
        else
        {
            auto parentInbox = inboxName + std::string({ listItem.delim });
            if (boost::istarts_with(utf8folderName.asString(), parentInbox))
            {
                auto res =
                    boost::ireplace_first_copy(utf8folderName.asString(), parentInbox, parentInbox);
                utf8folderName = Utf8MailboxName(res, listItem.delim);
            }
        }

        auto flags = listItem.flags;
        auto mailbox = std::make_shared<ImapListItem>(utf8folderName, flags);
        mailboxes.push_back(mailbox);
    }
    return std::make_shared<ImapList>(std::move(mailboxes));
}

//-----------------------------------------------------------------------------
// IMAP LIST command.
// Get delimeter:   A1 LIST "" ""
// Load folders:    A1 LIST "" *

class CommandList : public CommandBase<ImapList>
{
public:
    CommandList(const std::string& tag, const std::string& refName, const std::string& mailbox)
        : CommandBase(tag, "LIST", imap_quote(refName) + " " + imap_quote(mailbox))
    {
    }

    virtual grammar::ParseResultPtr parseResponse(std::shared_ptr<std::string> response)
    {
        std::vector<ListResponse> listResponses;
        grammar::ListResponseGrammar<std::string::iterator> listGrammar(
            [&](const ListResponse& listElement) { listResponses.push_back(listElement); });

        grammar::ParseResult res;
        bool parseOk =
            boost::spirit::qi::parse(response->begin(), response->end(), listGrammar, res);
        if (!parseOk) return nullptr;

        res.listResponses = std::move(listResponses);
        return std::make_shared<grammar::ParseResult>(std::move(res));
    }
};

} // namespace ymod_imap_client

#endif // _YMOD_IMAPCLIENT_COMMAND_LIST_H_
