#pragma once

#include "command.hpp"
#include <grammar/fetch_grammar.hpp>
#include <boost/algorithm/string.hpp>

namespace ymod_imap_client {

namespace {

void fillRange(std::vector<uint32_t>& outRange, const std::string& input)
{
    auto found = boost::algorithm::find(input, boost::algorithm::first_finder(":"));
    if (found)
    {
        auto firstNumRange = boost::make_iterator_range(input.begin(), found.begin());
        uint32_t start = boost::lexical_cast<uint32_t>(firstNumRange);

        auto secondNumRange = boost::make_iterator_range(found.end(), input.end());
        uint32_t end = boost::lexical_cast<uint32_t>(secondNumRange);

        for (uint32_t i = start; i <= end; ++i)
        {
            outRange.push_back(i);
        }
    }
    else
    {
        outRange.push_back(boost::lexical_cast<uint32_t>(input));
    }
}

}

template <>
std::shared_ptr<CopyuidResult> makeParsedResult(grammar::ParseResultPtr parsed)
{
    auto& uidplusResponse =
        ((parsed->uidplusResponse.uidvalidity) ? parsed->uidplusResponse :
                                                 parsed->taggedResponse.uidplusResponse);
    if (!uidplusResponse.uidvalidity)
    {
        return std::make_shared<CopyuidResult>();
    }

    std::vector<std::string> elements;
    boost::algorithm::split(elements, uidplusResponse.originalUids, boost::is_any_of(","));

    std::vector<uint32_t> oldUids;
    for (auto& elem : elements)
    {
        fillRange(oldUids, elem);
    }

    elements.clear();
    boost::algorithm::split(elements, uidplusResponse.newUids, boost::is_any_of(","));

    std::vector<uint32_t> newUids;
    for (auto& elem : elements)
    {
        fillRange(newUids, elem);
    }

    return std::make_shared<CopyuidResult>(
        uidplusResponse.uidvalidity, std::move(oldUids), std::move(newUids));
}

template <>
std::shared_ptr<AppenduidResult> makeParsedResult(grammar::ParseResultPtr parsed)
{
    auto& uidplusResponse =
        ((parsed->uidplusResponse.uidvalidity) ? parsed->uidplusResponse :
                                                 parsed->taggedResponse.uidplusResponse);
    if (!uidplusResponse.uidvalidity)
    {
        return std::make_shared<AppenduidResult>();
    }

    return std::make_shared<AppenduidResult>(
        uidplusResponse.uidvalidity, boost::lexical_cast<uint32_t>(uidplusResponse.newUids));
}

class CommandCreate : public CommandBase<ImapResult>
{
public:
    CommandCreate(const std::string& tag, const Utf8MailboxName& mailbox)
        : CommandBase(tag, "CREATE", imap_quote(Utf7ImapMailboxName(mailbox).asString()))
    {
    }
};

class CommandDelete : public CommandBase<ImapResult>
{
public:
    CommandDelete(const std::string& tag, const Utf8MailboxName& mailbox)
        : CommandBase(tag, "DELETE", imap_quote(Utf7ImapMailboxName(mailbox).asString()))
    {
    }
};

class CommandRename : public CommandBase<ImapResult>
{
public:
    CommandRename(
        const std::string& tag,
        const Utf8MailboxName& oldMailbox,
        const Utf8MailboxName& newMailbox)
        : CommandBase(
              tag,
              "RENAME",
              imap_quote(Utf7ImapMailboxName(oldMailbox).asString()) + " " +
                  imap_quote(Utf7ImapMailboxName(newMailbox).asString()))
    {
    }
};

class CommandExpunge : public CommandBase<ImapResult>
{
public:
    CommandExpunge(const std::string& tag) : CommandBase(tag, "EXPUNGE")
    {
    }
};

class CommandUidStore : public CommandBase<MessageSet>
{
public:
    CommandUidStore(
        const std::string& tag,
        const std::string& seqset,
        const std::string& storeType,
        const std::string& args)
        : CommandBase(tag, "UID STORE", seqset + " " + storeType + " (" + args + ")")
    {
    }

    virtual grammar::ParseResultPtr parseResponse(std::shared_ptr<std::string> response)
    {
        std::vector<FetchResponse> fetchResponses;
        grammar::FetchResponseGrammar<std::string::iterator> fetchGrammar(
            [&](const FetchResponse& fetchResponse) { fetchResponses.push_back(fetchResponse); });

        grammar::ParseResult res;
        bool parseOk =
            boost::spirit::qi::parse(response->begin(), response->end(), fetchGrammar, res);
        if (!parseOk) return nullptr;

        res.fetchResponses = std::move(fetchResponses);
        return std::make_shared<grammar::ParseResult>(std::move(res));
    }
};

class CommandUidMove : public CommandBase<CopyuidResult>
{
public:
    CommandUidMove(
        const std::string& tag,
        const std::string& seqset,
        const Utf8MailboxName& newMailbox)
        : CommandBase(
              tag,
              "UID MOVE",
              seqset + " " + imap_quote(Utf7ImapMailboxName(newMailbox).asString()))
    {
    }
};

class CommandUidCopy : public CommandBase<CopyuidResult>
{
public:
    CommandUidCopy(
        const std::string& tag,
        const std::string& seqset,
        const Utf8MailboxName& newMailbox)
        : CommandBase(
              tag,
              "UID COPY",
              seqset + " " + imap_quote(Utf7ImapMailboxName(newMailbox).asString()))
    {
    }
};

class CommandAppend : public CommandBase<AppenduidResult>
{
public:
    CommandAppend(
        const std::string& tag,
        std::string&& msg_body,
        const Utf8MailboxName& mailbox,
        const std::string& flags,
        const std::string& date)
        : CommandBase(tag, "APPEND"), body(std::forward<std::string>(msg_body))
    {
        customArgs = imap_quote(Utf7ImapMailboxName(mailbox).asString());
        if (!flags.empty())
        {
            customArgs += " (" + flags + ")";
        }

        if (!date.empty())
        {
            customArgs += " " + date;
        }

        customArgs += " {" + boost::lexical_cast<std::string>(body.size()) + "}";
    }

    virtual const string& args() const
    {
        return customArgs;
    }

    virtual ImapFilterState filterState() const
    {
        return ImapFilterState::EXPECT_COMMAND_CONTINUATION;
    }

    virtual CommandState handleResponse(std::shared_ptr<std::string> response)
    {
        if ((*response)[0] == '+')
        {
            body.append("\r\n");
            return SendData{ std::move(body), ImapFilterState::EXPECT_RESPONSE_DONE };
        }
        return CommandBase<AppenduidResult>::handleResponse(response);
    }

private:
    std::string body;
    std::string customArgs;
};

} // namespace ymod_imap_client
