#ifndef _YMOD_IMAP_CLIENT_GRAMMAR_FETCH_RESPONSE_H_
#define _YMOD_IMAP_CLIENT_GRAMMAR_FETCH_RESPONSE_H_

#include "imap_base.hpp"

#include <boost/variant.hpp>
#include <boost/optional.hpp>
#include <boost/algorithm/string.hpp>

namespace ymod_imap_client { namespace grammar {

static const int start_year = 1900;

#define CONVERT_OFFSET_TO_SECONDS(offset) (offset) / 100 * 3600 + (offset) % 100 * 60

struct NumTag;
struct UidTag;
struct BodyTag;
struct FlagsTag;
struct BodyTailSizeTag;
struct XGMLabelTag;
struct SizeTag;

using Num = TypeWrap<uint32_t, NumTag>;
using Uid = TypeWrap<uint32_t, UidTag>;
using Flags = TypeWrap<uint32_t, FlagsTag>;
using BodyTailSize = TypeWrap<uint32_t, BodyTailSizeTag>;
using XGMLabel = TypeWrap<std::string, XGMLabelTag>;
using XGMLabels = std::vector<XGMLabel>;
using Time = std::tm;
using Size = TypeWrap<uint64_t, SizeTag>;

template <typename Iterator>
using Body = TypeWrap<Range<Iterator>, BodyTag>;

template <typename Iterator>
using FetchAttr =
    boost::variant<Num, Uid, Flags, Body<Iterator>, BodyTailSize, XGMLabels, Time, Size>;

template <typename Iterator>
using FetchAttrs = std::vector<FetchAttr<Iterator>>;

template <typename Iterator>
struct FetchResponseFactory : public boost::static_visitor<>
{
    using Type = FetchResponseFactory<Iterator>;

    FetchResponseFactory(const FetchAttrs<Iterator>& attrs)
    {
        for (auto& attr : attrs)
        {
            boost::apply_visitor(*this, attr);
        }
    }

    FetchResponse produce() const
    {
        return product;
    }

    void operator()(const Num& num) const
    {
        product.num = static_cast<uint32_t>(num);
    }
    void operator()(const Uid& uid) const
    {
        product.uid = static_cast<uint32_t>(uid);
    }
    void operator()(const Flags& flags) const
    {
        product.flags = static_cast<uint32_t>(flags);
    }
    void operator()(const Size& size) const
    {
        product.size = static_cast<uint64_t>(size);
    }
    void operator()(const Time& time) const
    {
        product.internaldate = static_cast<std::tm>(time);
    }
    void operator()(const BodyTailSize& /* tailSize */) const
    { /* product.flags = static_cast<uint32_t>(flags);*/
    }
    void operator()(const XGMLabels& xgmlabels) const
    {
        for (auto xgmlLabel : xgmlabels)
        {
            product.xgmlabels.emplace_back(
                Utf8Label(Utf7ImapLabel::create(static_cast<std::string>(xgmlLabel))));
        }
    }

    void operator()(const Body<Iterator>& body) const
    {
        auto bodyRange = body.value;
        product.body = std::make_shared<std::string>(bodyRange.begin(), bodyRange.end());
    }

    mutable FetchResponse product;
};

template <typename Iterator>
struct ResponseFactory<Iterator, FetchResponse>
{
    using Type = typename FetchResponseFactory<Iterator>::Type;
};

template <class Iterator>
struct FetchResponseBaseGrammar : public ImapBasicGrammar<Iterator>
{

    using ImapGrammar = ImapBasicGrammar<Iterator>;

    FetchResponseBaseGrammar()
    {
        message_data = (fetch_prefix_att(bph::ref(bsq::_val)) >> msg_att);

        fetch_prefix_att = (msg_num >> omit[" FETCH "])[bph::push_back(bsq::_r1, bsq::_1)];
        msg_num = bs::uint_;
        msg_att = "(" >> ((msg_att_dynamic | msg_att_static) % " ") >> ")";
        msg_att_dynamic = flags_list_proxy;
        msg_att_static = uniqueid | xgm_labels | internaldate | size | body;

        flags_list_proxy = "FLAGS (" >> -flags_list >> ")";
        flags_list = (flag_fetch(bsq::_val) >> *(" " >> flag_fetch(bsq::_val)))[bsq::_val];
        flag_fetch = bsq::no_case[bs::lit("\\Recent")[bsq::_r1 |= FetchResponse::mf_recent]] |
            flag(bsq::_r1);

        flag = bsq::no_case
                   [bs::lit("\\Answered")[bsq::_r1 |= FetchResponse::mf_answered] |
                    bs::lit("\\Flagged")[bsq::_r1 |= FetchResponse::mf_flagged] |
                    bs::lit("\\Deleted")[bsq::_r1 |= FetchResponse::mf_deleted] |
                    bs::lit("\\Seen")[bsq::_r1 |= FetchResponse::mf_seen] |
                    bs::lit("\\Draft")[bsq::_r1 |= FetchResponse::mf_draft]] |
            "\\" >> +ImapGrammar::ATOM_CHAR | +ImapGrammar::ATOM_CHAR;

        uniqueid = "UID " >> bs::uint_;
        xgm_labels = "X-GM-LABELS (" >> -(xgm_label % " ") >> ")";
        xgm_label = ImapGrammar::quoted | ImapGrammar::astring_;
        quoted_xgm_label = bsq::omit['"'] >> (*ImapGrammar::QUOTED_CHAR) >> bsq::omit['"'];
        body = "BODY[] " >> ImapGrammar::literal;

        internaldate = omit["INTERNALDATE "] >> time;
        time = omit["\""] >> day[bph::bind(&std::tm::tm_mday, _val) = bsq::_1] >> "-" >>
            month[bph::bind(&std::tm::tm_mon, _val) = bsq::_1] >> "-" >>
            year[bph::bind(&std::tm::tm_year, _val) = bsq::_1 - start_year] >> " " >>
            hour[bph::bind(&std::tm::tm_hour, _val) = bsq::_1] >> ":" >>
            minute[bph::bind(&std::tm::tm_min, _val) = bsq::_1] >> ":" >>
            sec[bph::bind(&std::tm::tm_sec, _val) = bsq::_1] >> " " >>
            offset[bph::bind(&std::tm::tm_gmtoff, _val) = CONVERT_OFFSET_TO_SECONDS(bsq::_1)] >>
            omit["\""];

        day = bsq::uint_;
        month = bsq::no_case
            [bs::lit("Jan")[bsq::_val = 0] | bs::lit("Feb")[bsq::_val = 1] |
             bs::lit("Mar")[bsq::_val = 2] | bs::lit("Apr")[bsq::_val = 3] |
             bs::lit("May")[bsq::_val = 4] | bs::lit("Jun")[bsq::_val = 5] |
             bs::lit("Jul")[bsq::_val = 6] | bs::lit("Aug")[bsq::_val = 7] |
             bs::lit("Sep")[bsq::_val = 8] | bs::lit("Oct")[bsq::_val = 9] |
             bs::lit("Nov")[bsq::_val = 10] | bs::lit("Dec")[bsq::_val = 11]];
        year = bsq::uint_;
        hour = bsq::uint_;
        minute = bsq::uint_;
        sec = bsq::uint_;
        offset = -omit[lit("+")] >> bsq::int_;

        size = "RFC822.SIZE " >> bs::ulong_long;
    }

    bsq::rule<Iterator, FetchAttrs<Iterator>()> message_data;

    bsq::rule<Iterator, void(FetchAttrs<Iterator>&)> fetch_prefix_att;
    bsq::rule<Iterator, Num()> msg_num;

    bsq::rule<Iterator, FetchAttrs<Iterator>()> msg_att;
    bsq::rule<Iterator, FetchAttr<Iterator>()> msg_att_static;
    bsq::rule<Iterator, FetchAttr<Iterator>()> msg_att_dynamic;

    bsq::rule<Iterator, Flags()> flags_list_proxy;
    bsq::rule<Iterator, uint32_t()> flags_list;
    bsq::rule<Iterator, void(uint32_t&)> flag_fetch;
    bsq::rule<Iterator, void(uint32_t&)> flag;

    bsq::rule<Iterator, Uid()> uniqueid;
    bsq::rule<Iterator, XGMLabels()> xgm_labels;
    bsq::rule<Iterator, XGMLabel()> xgm_label;
    bsq::rule<Iterator, Body<Iterator>()> body;

    bsq::rule<Iterator, std::string()> quoted_xgm_label;
    bsq::rule<Iterator, Time()> internaldate;
    bsq::rule<Iterator, std::tm()> time;
    bsq::rule<Iterator, Size()> size;

    boost::spirit::qi::rule<Iterator, int()> year;
    boost::spirit::qi::rule<Iterator, int()> month;
    boost::spirit::qi::rule<Iterator, int()> day;
    boost::spirit::qi::rule<Iterator, int()> hour;
    boost::spirit::qi::rule<Iterator, int()> minute;
    boost::spirit::qi::rule<Iterator, int()> sec;
    boost::spirit::qi::rule<Iterator, long()> offset;
};

template <class Iterator>
struct FetchResponseGrammar
    : public bsq::grammar<Iterator, ParseResult()>
    , public FetchResponseBaseGrammar<Iterator>
{
    using ParseHandler = typename ParseHandlerWrapper<Iterator, FetchResponse>::Type;
    using ParseCallback = typename ParseHandler::Callback;

    FetchResponseGrammar(ParseCallback h)
        : FetchResponseGrammar::base_type(this->fullResponse), parseHandler(ParseHandler(h))
    {
        this->untaggedResponses =
            *(("* " >> !this->resp_cond_bye >>
               (this->message_data[parseHandler(bsq::_1)] | omit[this->reason]) >>
               this->imapEol)[parseHandler()]);
    }

    boost::phoenix::function<ParseHandler> parseHandler;
};

//-----------------------------------------------------------------------------
// Parser for FETCH BODY[] response prefix like: * 1 FETCH (BODY[] {2217}\r\n

template <class Iterator>
struct FetchBodyPrefixFactory : public boost::static_visitor<>
{
    using Type = FetchBodyPrefixFactory<Iterator>;

    FetchBodyPrefixFactory(const FetchAttrs<Iterator>& attrs)
    {
        for (auto& attr : attrs)
        {
            boost::apply_visitor(*this, attr);
        }
    }

    template <typename T>
    void operator()(const T&) const
    {
    }
    void operator()(const BodyTailSize& size) const
    {
        product.size = static_cast<uint32_t>(size);
    }

    FetchBodyPrefixResponse produce() const
    {
        return product;
    }
    mutable FetchBodyPrefixResponse product;
};

template <typename Iterator>
struct ResponseFactory<Iterator, FetchBodyPrefixResponse>
{
    using Type = typename FetchBodyPrefixFactory<Iterator>::Type;
};

template <class Iterator>
struct FetchBodyPrefixGrammar
    : public bsq::grammar<Iterator>
    , public FetchResponseBaseGrammar<Iterator>
{
    using ParseHandler = typename ParseHandlerWrapper<Iterator, FetchBodyPrefixResponse>::Type;
    using ParseCallback = typename ParseHandler::Callback;

    FetchBodyPrefixGrammar(ParseCallback callback)
        : FetchBodyPrefixGrammar::base_type(partialResponse), parseHandler(ParseHandler(callback))
    {
        partialResponse = omit[this->untaggedResponses] >> response_message_data[parseHandler()] |
            (this->fullResponse);

        this->untaggedResponses =
            *("* " >> !(this->resp_cond_bye | this->message_data) >> (omit[this->reason]) >>
              this->imapEol);

        response_message_data = bs::lit("* ") >> this->message_data[parseHandler(bsq::_1)];

        this->msg_att =
            "(" >> *((this->msg_att_dynamic | this->msg_att_static) >> " ") >> body_prefix
            // Do not include closing ")"
            ;

        body_prefix = "BODY[] " >> ('{' >> uint_ >> '}' >> this->imapEol);
    }

    bsq::rule<Iterator> partialResponse;
    bsq::rule<Iterator> response_message_data;
    boost::spirit::qi::rule<Iterator, BodyTailSize()> body_prefix;

    boost::phoenix::function<ParseHandler> parseHandler;
};

} // namespace grammar
} // namespace ymod_imap_client

#endif // _YMOD_IMAP_CLIENT_GRAMMAR_FETCH_RESPONSE_H_
