#ifndef _YMOD_IMAP_CLIENT_GRAMMAR_IMAP_BASE_H_
#define _YMOD_IMAP_CLIENT_GRAMMAR_IMAP_BASE_H_

#include <ymod_imapclient/imap_response.h>

#include <boost/spirit/include/qi.hpp>
#include <boost/spirit/include/phoenix.hpp>
#include <boost/spirit/include/phoenix_fusion.hpp>
#include <boost/fusion/include/boost_tuple.hpp>
#include <boost/lambda/lambda.hpp>
#include <boost/range/iterator_range.hpp>
#include <boost/spirit/include/qi_no_case.hpp>

namespace ymod_imap_client { namespace grammar {

namespace bs = boost::spirit;
namespace bsq = boost::spirit::qi;
namespace bph = boost::phoenix;

using boost::spirit::qi::char_;
using boost::spirit::lit;
using boost::spirit::uint_;
using boost::spirit::omit;
using boost::spirit::raw;
using boost::spirit::qi::repeat;
using boost::spirit::qi::_r1;
using boost::spirit::qi::_a;
using boost::spirit::qi::no_case;
using bsq::_val;

enum class ResponseStatus
{
    OK,
    NO,
    BAD,
    Fatal
};

struct UidplusResponse
{
    uint32_t uidvalidity = 0;
    std::string originalUids;
    std::string newUids;

    bool operator==(const UidplusResponse& rhs) const
    {
        return uidvalidity == rhs.uidvalidity && originalUids == rhs.originalUids &&
            newUids == rhs.newUids;
    }
};

struct TaggedResponse
{
    ResponseStatus status = ResponseStatus::OK;
    std::string reason;
    std::string code;

    UidplusResponse uidplusResponse;

    bool operator==(const TaggedResponse& rhs) const
    {
        return status == rhs.status && reason == rhs.reason && code == rhs.code &&
            uidplusResponse == rhs.uidplusResponse;
    }
};

struct StatusResponse
{
    std::string mailboxName;
    MailboxInfoResponse mailboxInfoResponse;

    bool operator==(const StatusResponse& rhs) const
    {
        return mailboxName == rhs.mailboxName && mailboxInfoResponse == rhs.mailboxInfoResponse;
    }
};

struct ParseResult
{
    std::vector<FetchResponse> fetchResponses;
    std::vector<ListResponse> listResponses;

    StatusResponse statusResponse;
    MailboxInfoResponse mailboxInfoResponse;

    std::vector<std::string> idData;
    std::vector<std::string> capabilityData;

    TaggedResponse taggedResponse;
    UidplusResponse uidplusResponse;

    operator bool() const
    {
        return taggedResponse.status == ResponseStatus::OK;
    }
    bool operator==(const ParseResult& rhs) const
    {
        return fetchResponses == rhs.fetchResponses && listResponses == rhs.listResponses &&
            mailboxInfoResponse == rhs.mailboxInfoResponse && idData == rhs.idData &&
            capabilityData == rhs.capabilityData && taggedResponse == rhs.taggedResponse &&
            uidplusResponse == rhs.uidplusResponse;
    }
};

using ParseResultPtr = std::shared_ptr<ParseResult>;

template <class Iterator>
using Range = boost::iterator_range<Iterator>;

template <class Iterator>
struct ImapBasicGrammar
{
    ImapBasicGrammar()
    {
        fullResponse = untaggedResponses[_val = bsq::_1] >>
            (responseTagged[bph::bind(&ParseResult::taggedResponse, _val) = bsq::_1] |
             responseFatal[bph::bind(&ParseResult::taggedResponse, _val) = bsq::_1]);

        untaggedResponses = *(
            "* " >> !resp_cond_bye >>
            (capabilityResponse[bph::bind(&ParseResult::capabilityData, _val) = bsq::_1] |
             idResponse[bph::bind(&ParseResult::idData, _val) = bsq::_1] |
             fullStatusResponse[bph::bind(&ParseResult::statusResponse, _val) = bsq::_1] |
             "OK " >> resp_uidplus_full[bph::bind(&ParseResult::uidplusResponse, _val) = bsq::_1] >>
                 -omit[reason] |
             omit[reason]) >>
            imapEol);

        responseTagged = omit[tag >> char(' ')] >> resp_cond_state >> imapEol;

        responseFatal = lit("* ") >>
            resp_cond_bye[bph::bind(&TaggedResponse::reason, _val) = bsq::_1] >>
            imapEol[bph::bind(&TaggedResponse::status, _val) = ResponseStatus::Fatal];

        capabilityResponse = "CAPABILITY" >> *(" " >> capability);
        capability = +(char_ - ' ' - bsq::eol);

        idResponse = "ID (" >> quoted >> *(" " >> quoted) >> ")";

        fullStatusResponse = "STATUS " >>
            astring_[bph::bind(&StatusResponse::mailboxName, _val) = bsq::_1] >> " (" >>
            (statusResponse(bph::bind(&StatusResponse::mailboxInfoResponse, _val)) % " ") >> ")" >>
            *omit[" "];

        statusResponse =
            "MESSAGES " >> uint_[bph::bind(&MailboxInfoResponse::exists, _r1) = bsq::_1] |
            "RECENT " >> uint_[bph::bind(&MailboxInfoResponse::recent, _r1) = bsq::_1] |
            "UIDNEXT " >> uint_[bph::bind(&MailboxInfoResponse::uidnext, _r1) = bsq::_1] |
            "UIDVALIDITY " >> uint_[bph::bind(&MailboxInfoResponse::uidvalidity, _r1) = bsq::_1] |
            "UNSEEN " >> uint_[bph::bind(&MailboxInfoResponse::unseen, _r1) = bsq::_1];

        resp_cond_auth = lit("OK ") >> reason;
        resp_cond_bye = lit("BYE ") >> reason;

        reason = *(char_ - bsq::eol);

        resp_cond_state =
            (lit("OK ")[bph::bind(&TaggedResponse::status, _val) = ResponseStatus::OK] |
             lit("NO ")[bph::bind(&TaggedResponse::status, _val) = ResponseStatus::NO] |
             lit("BAD ")[bph::bind(&TaggedResponse::status, _val) = ResponseStatus::BAD]) >>
            -(resp_text_code[bph::bind(&TaggedResponse::code, _val) = bsq::_1] |
              resp_uidplus_full[bph::bind(&TaggedResponse::uidplusResponse, _val) = bsq::_1]) >>
            reason[bph::bind(&TaggedResponse::reason, _val) = bsq::_1];

        // According TO https://tools.ietf.org/html/rfc5530#section-4
        resp_text_code = "[" >> (+ATOM_CHAR | string_) >> "]" >> -lit(" ");

        resp_uidplus_full = "[" >> resp_uidplus >> "]" >> -lit(" ");
        resp_uidplus = ("APPENDUID " >> resp_appenduid) | ("COPYUID " >> resp_copyuid);
        resp_appenduid = uint_[bph::bind(&UidplusResponse::uidvalidity, _val) = bsq::_1] >> " " >>
            ATOM_STRING[bph::bind(&UidplusResponse::newUids, _val) = bsq::_1];
        resp_copyuid = uint_[bph::bind(&UidplusResponse::uidvalidity, _val) = bsq::_1] >> " " >>
            ATOM_STRING[bph::bind(&UidplusResponse::originalUids, _val) = bsq::_1] >> " " >>
            ATOM_STRING[bph::bind(&UidplusResponse::newUids, _val) = bsq::_1];

        imapEol = "\r\n";
        tag = +(ASTRING_CHAR - char_("+*"));

        QUOTED_CHAR = (char_ - quoted_specials) | (omit['\\'] >> quoted_specials);

        quoted_specials = char_("\"\\");

        astring_ = (+ASTRING_CHAR) | string_;

        ASTRING_CHAR = ATOM_CHAR | ']';

        ATOM_CHAR = (char_ - atom_specials);
        ATOM_STRING = +ATOM_CHAR;

        atom_specials = char_("(){ %*]") | char_(0x01, 0x1F) | quoted_specials;

        string_ = quoted | literal;

        quoted = omit['"'] >> *QUOTED_CHAR >> omit['"'];

        literal = omit['{' >> uint_[_a = bsq::_1] >> '}' >> imapEol] >> raw[repeat(_a)[char_]];

        command_continuation =
            lit("+")[bph::bind(&TaggedResponse::status, _val) = ResponseStatus::OK] >>
            -(" " >> -reason[bph::bind(&TaggedResponse::reason, _val)]) >> imapEol;
    }

    boost::spirit::qi::rule<Iterator, ParseResult()> fullResponse;

    boost::spirit::qi::rule<Iterator, ParseResult()> untaggedResponses;
    boost::spirit::qi::rule<Iterator, TaggedResponse()> responseTagged;
    boost::spirit::qi::rule<Iterator, TaggedResponse()> responseFatal;

    boost::spirit::qi::rule<Iterator, std::string()> capability;
    boost::spirit::qi::rule<Iterator, std::vector<std::string>()> capabilityResponse;
    boost::spirit::qi::rule<Iterator, std::vector<std::string>()> idResponse;
    boost::spirit::qi::rule<Iterator, StatusResponse()> fullStatusResponse;
    boost::spirit::qi::rule<Iterator, void(MailboxInfoResponse&)> statusResponse;

    boost::spirit::qi::rule<Iterator, void(MailboxInfoResponse&)> messagesResponse;
    boost::spirit::qi::rule<Iterator, void(MailboxInfoResponse&)> recentResponse;
    boost::spirit::qi::rule<Iterator, void(MailboxInfoResponse&)> uidnextResponse;
    boost::spirit::qi::rule<Iterator, void(MailboxInfoResponse&)> uidvalidityResponse;
    boost::spirit::qi::rule<Iterator, void(MailboxInfoResponse&)> unseenResponse;

    boost::spirit::qi::rule<Iterator, std::string()> resp_cond_auth;
    boost::spirit::qi::rule<Iterator, std::string()> resp_cond_bye;
    boost::spirit::qi::rule<Iterator, std::string()> resp_text_code;
    boost::spirit::qi::rule<Iterator, UidplusResponse()> resp_uidplus_full;
    boost::spirit::qi::rule<Iterator, UidplusResponse()> resp_uidplus;
    boost::spirit::qi::rule<Iterator, UidplusResponse()> resp_appenduid;
    boost::spirit::qi::rule<Iterator, UidplusResponse()> resp_copyuid;

    boost::spirit::qi::rule<Iterator, std::string()> reason;

    boost::spirit::qi::rule<Iterator> imapEol;
    boost::spirit::qi::rule<Iterator> tag;
    boost::spirit::qi::rule<Iterator, TaggedResponse()> resp_cond_state;

    boost::spirit::qi::rule<Iterator, char()> QUOTED_CHAR;
    boost::spirit::qi::rule<Iterator, char()> quoted_specials;
    boost::spirit::qi::rule<Iterator, std::string()> astring_;
    boost::spirit::qi::rule<Iterator, char()> ASTRING_CHAR;
    boost::spirit::qi::rule<Iterator, char()> ATOM_CHAR;
    boost::spirit::qi::rule<Iterator, std::string()> ATOM_STRING;
    boost::spirit::qi::rule<Iterator, char()> atom_specials;
    boost::spirit::qi::rule<Iterator, std::string()> string_;
    boost::spirit::qi::rule<Iterator, std::string()> quoted;
    boost::spirit::qi::rule<Iterator, Range<Iterator>(), boost::spirit::qi::locals<std::size_t>>
        literal;

    boost::spirit::qi::rule<Iterator, TaggedResponse()> command_continuation;
};

//-----------------------------------------------------------------------------
// Grammar for connect server greetings

template <class Iterator>
struct Greeting
    : public bsq::grammar<Iterator, TaggedResponse()>
    , public ImapBasicGrammar<Iterator>
{
    using ImapGrammar = ImapBasicGrammar<Iterator>;

    Greeting() : Greeting::base_type(greeting)
    {
        auto valState = bph::bind(&TaggedResponse::status, _val);
        auto valReason = bph::bind(&TaggedResponse::reason, _val);
        greeting = lit("* ") >>
            (ImapGrammar::resp_cond_auth[valState = ResponseStatus::OK, valReason = bsq::_1] |
             ImapGrammar::resp_cond_bye[valState = ResponseStatus::Fatal, valReason = bsq::_1]) >>
            this->imapEol;
    }

    boost::spirit::qi::rule<Iterator, TaggedResponse()> greeting;
};

//-----------------------------------------------------------------------------
// Grammar for response ending

template <class Iterator>
struct ResponseDone
    : public ImapBasicGrammar<Iterator>
    , public bsq::grammar<Iterator, ParseResult()>
{
    ResponseDone() : ResponseDone::base_type(this->fullResponse)
    {
    }
};

template <class Iterator>
struct ResponseBye
    : public ImapBasicGrammar<Iterator>
    , public bsq::grammar<Iterator, ParseResult()>
{
    ResponseBye() : ResponseBye::base_type(this->fullResponse)
    {
        this->fullResponse = this->untaggedResponses[_val = bsq::_1] >>
            (this->responseTagged[bph::bind(&ParseResult::taggedResponse, _val) = bsq::_1] |
             this->responseBye[bph::bind(&ParseResult::taggedResponse, _val) = bsq::_1]);

        responseBye = lit("* ") >>
            this->resp_cond_bye[bph::bind(&TaggedResponse::reason, _val) = bsq::_1] >>
            this->imapEol[bph::bind(&TaggedResponse::status, _val) = ResponseStatus::OK];
    }

    boost::spirit::qi::rule<Iterator, TaggedResponse()> responseBye;
};

//-----------------------------------------------------------------------------
// Grammar for literal command continuation. literal {+}

template <class Iterator>
struct CommandContinuation
    : public ImapBasicGrammar<Iterator>
    , public bsq::grammar<Iterator, TaggedResponse()>
{
    CommandContinuation() : CommandContinuation::base_type(this->command_continuation)
    {
    }
};

//-----------------------------------------------------------------------------
// Common auxiliary types

template <class Iterator, class ResponseType>
struct ResponseFactory;

template <class Iterator, class Result>
struct ParseHandlerWrapper
{
    using ResultFactory = typename ResponseFactory<Iterator, Result>::Type;
    using Callback = std::function<void(const Result&)>;
    using Type = ParseHandlerWrapper<Iterator, Result>;

    Callback callback;
    ParseHandlerWrapper(Callback callback) : callback(callback)
    {
    }

    template <typename T>
    struct result
    {
        typedef void type;
    };

    void operator()(const ResultFactory& factory) const
    {
        *res = factory.produce();
        *hasData = true;
    }

    void operator()() const
    {
        if (*hasData)
        {
            callback(*res);
            *hasData = false;
        }
    }

    std::shared_ptr<Result> res = std::make_shared<Result>();
    std::shared_ptr<bool> hasData = std::make_shared<bool>(false);
};

template <typename BaseType, typename Tag>
struct TypeWrap
{
    TypeWrap() = default;
    TypeWrap(const TypeWrap&) = default;
    TypeWrap(const BaseType& val) : value(val)
    {
    }

    template <class Other>
    TypeWrap& operator=(const Other& rhs)
    {
        value = rhs;
        return *this;
    }

    operator BaseType() const
    {
        return value;
    }

    BaseType value;
};

// Boost.Spirit debug purpouses
template <typename Stream, typename WrappedType, typename WrapperTag>
Stream& operator<<(Stream&& stream, const TypeWrap<WrappedType, WrapperTag>& wrapper)
{
    stream << static_cast<WrappedType>(wrapper);
    return stream;
}

template <typename Stream>
Stream& operator<<(Stream&& stream, const grammar::TaggedResponse& resp)
{
    stream << "[" << resp.code << "]" << resp.reason;
    return stream;
}

template <typename Stream>
Stream& operator<<(Stream&& stream, const grammar::UidplusResponse& resp)
{
    stream << "Uidval: " << resp.uidvalidity << ", Old uids: " << resp.originalUids
           << ", New uids: " << resp.newUids;
    return stream;
}

} // namespace grammar
} // namespace ymod_imap_client

#endif // _YMOD_IMAP_CLIENT_GRAMMAR_IMAP_BASE_H_
