#include "imap_client_impl.h"

#include <yplatform/find.h>
#include <boost/algorithm/string/join.hpp>
#include <iostream>

namespace ymod_imap_client {

const std::string& convert_fetch_arg_to_string(FetchArg arg)
{
    static constexpr std::initializer_list<const char*> args = { "UID",          "FLAGS",
                                                                 "INTERNALDATE", "RFC822.SIZE",
                                                                 "BODY.PEEK[]",

                                                                 "X-GM-LABELS" };

    static_assert(
        args.size() == static_cast<size_t>(FetchArg::COUNT),
        "FetchArgs count doesn't correspond with theirs string names count");
    static const std::vector<std::string> string_args(args.begin(), args.end());

    return string_args[static_cast<int>(arg)];
}

std::string convert_fetch_args_to_string(FetchArgs args)
{
    std::vector<std::string> string_args;
    string_args.reserve(args.size());
    std::transform(
        args.begin(), args.end(), std::back_inserter(string_args), convert_fetch_arg_to_string);

    return boost::algorithm::join(string_args, " ");
}

ImapClientImpl::~ImapClientImpl()
{
    context->erase_module_data("imap_client");
}

FutureConnectResult ImapClientImpl::connect(const string& server, unsigned short port, bool ssl)
{
    auto serverLoggingAlias = server;
    return session->connect(server, serverLoggingAlias, port, ssl);
}

FutureConnectResult ImapClientImpl::connect(
    const string& server,
    const string& serverLoggingAlias,
    unsigned short port,
    bool ssl)
{
    return session->connect(server, serverLoggingAlias, port, ssl);
}

FutureImapResult ImapClientImpl::login(const string& user, const string& password)
{
    return session->login(user, password);
}

FutureImapResult ImapClientImpl::loginOauth(
    const string& user,
    const string& accessToken,
    OauthLoginType type)
{
    return session->loginOauth(user, accessToken, type);
}

FutureImapResult ImapClientImpl::authPlain(const string& user, const string& password)
{
    return session->authPlain(user, password);
}

FutureCapability ImapClientImpl::capability()
{
    return session->capability();
}

FutureIdResult ImapClientImpl::id(const string& idData)
{
    return session->id(idData.empty() ? *defaultIdInfo : idData);
}

FutureImapResult ImapClientImpl::startTls()
{
    return session->startTls();
}

FutureImapList ImapClientImpl::list(const std::string& refName, const std::string& mailbox)
{
    return session->list(refName, mailbox);
}

FutureImapStatus ImapClientImpl::status(const Utf8MailboxName& mailbox, const std::string& fields)
{
    return session->status(mailbox, fields);
}

FutureImapMailbox ImapClientImpl::examine(const Utf8MailboxName& mailbox)
{
    return session->examine(mailbox);
}

FutureImapResult ImapClientImpl::select(const Utf8MailboxName& mailbox)
{
    return session->select(mailbox);
}

FutureMessageSet ImapClientImpl::fetch(const string& seqset, const FetchArgs& args)
{
    return session->fetch(seqset, convert_fetch_args_to_string(args));
}

FutureMessageSet ImapClientImpl::uidFetch(const string& seqset, const FetchArgs& args)
{
    return session->uidFetch(seqset, convert_fetch_args_to_string(args));
}

FutureImapResult ImapClientImpl::noop()
{
    return session->noop();
}

FutureImapResult ImapClientImpl::logout()
{
    return session->logout();
}

FutureImapResult ImapClientImpl::createMailbox(const Utf8MailboxName& mailbox)
{
    return session->createMailbox(mailbox);
}

FutureImapResult ImapClientImpl::deleteMailbox(const Utf8MailboxName& mailbox)
{
    return session->deleteMailbox(mailbox);
}

FutureImapResult ImapClientImpl::renameMailbox(
    const Utf8MailboxName& oldMailbox,
    const Utf8MailboxName& newMailbox)
{
    return session->renameMailbox(oldMailbox, newMailbox);
}

FutureImapResult ImapClientImpl::expunge()
{
    return session->expunge();
}

FutureMessageSet ImapClientImpl::uidStore(
    const std::string& seqset,
    const std::string& storeType,
    const std::string& args)
{
    return session->uidStore(seqset, storeType, args);
}

FutureCopyuidResult ImapClientImpl::uidMove(
    const std::string& seqset,
    const Utf8MailboxName& newMailbox)
{
    return session->uidMove(seqset, newMailbox);
}

FutureCopyuidResult ImapClientImpl::uidCopy(
    const std::string& seqset,
    const Utf8MailboxName& newMailbox)
{
    return session->uidCopy(seqset, newMailbox);
}

FutureAppenduidResult ImapClientImpl::append(
    std::string&& body,
    const Utf8MailboxName& mailbox,
    const std::string& flags,
    const std::string& date)
{
    return session->append(std::forward<std::string>(body), mailbox, flags, date);
}

string ImapClientImpl::serverIp() const
{
    return session->remote_addr().to_string();
}

void ImapClientImpl::setVerbose(const VerbositySettings& verbosity)
{
    return session->set_verbose(verbosity);
}

Statistics ImapClientImpl::getStats()
{
    return session->getStats();
}

void ImapClientImpl::setCommandTimeout(const yplatform::time_traits::duration& timeout)
{
    session->setCommandTimeout(timeout);
}

} // namespace ymod_imap_client
