#include <ymod_imapclient/imap_result.h>
#include <regex>
#include <iostream>

namespace ymod_imap_client {

Utf8MailboxName::Utf8MailboxName() : std::string()
{
}

Utf8MailboxName::Utf8MailboxName(const std::string& name, char delim)
    : std::string(name), delim(delim)
{
}

Utf8MailboxName::Utf8MailboxName(const Utf7ImapMailboxName& utf7Name)
    : std::string(folderNameFromUtf7Imap(utf7Name.asString(), utf7Name.delim))
    , delim(utf7Name.delim)
{
}

bool Utf8MailboxName::isParentOf(const Utf8MailboxName& other) const
{
    if (size() + 1 >= other.size()) return false;

    std::string prefix = *this + delim;
    auto prefixMatch = std::mismatch(begin(), end(), other.begin());
    return end() == prefixMatch.first && delim == *prefixMatch.second;
}

Utf7ImapMailboxName::Utf7ImapMailboxName(const std::string& name, char delim)
    : std::string(name), delim(delim)
{
}

Utf7ImapMailboxName::Utf7ImapMailboxName(const Utf8MailboxName& utf8Name)
    : std::string(folderNameToUtf7Imap(utf8Name.asString(), utf8Name.delim)), delim(utf8Name.delim)
{
}

//-----------------------------------------------------------------------------
// Utf7/Utf8 labels

Utf8Label::Utf8Label(const Utf7ImapLabel& utf7Label)
    : std::string(utf7imapToUtf8(utf7Label.asString()))
{
}

//-----------------------------------------------------------------------------
// List results

ImapListItem::ImapListItem(const Utf8MailboxName& name, uint32_t flags) : name(name), flags(flags)
{
}

//-----------------------------------------------------------------------------
// Imap capabilities

ImapCapability::ImapCapability(Capability&& capability) : ImapResult(true), capability(capability)
{
}

//-----------------------------------------------------------------------------
// Result of LIST/LSUB/XLIST command

ImapList::ImapList(ImapMailboxList&& mailboxes) : ImapResult(true), mailboxes(mailboxes)
{
}

//-----------------------------------------------------------------------------
// Result of EXAMINE/SELECT command

ImapMailboxResult::ImapMailboxResult(MailboxInfo&& mailboxInfo)
    : ImapResult(true), mailboxInfo(mailboxInfo)
{
}

ImapStatus::ImapStatus(MailboxInfo&& mailboxInfo) : ImapResult(true), mailboxInfo(mailboxInfo)
{
}

//-----------------------------------------------------------------------------
// Message UID set

MessageSet::MessageSet(ImapMessageVector&& messages) : ImapResult(true), messages(messages)
{
}

IdResult::IdResult(std::map<std::string, std::string>&& data) : ImapResult(true), data(data)
{
}

CopyuidResult::CopyuidResult(
    uint32_t uidvalidity,
    std::vector<uint32_t>&& oldUids,
    std::vector<uint32_t>&& newUids)
    : ImapResult(true)
    , uidvalidity(uidvalidity)
    , oldUids(std::forward<std::vector<uint32_t>>(oldUids))
    , newUids(std::forward<std::vector<uint32_t>>(newUids))
{
}

AppenduidResult::AppenduidResult(uint32_t uidvalidity, uint32_t uid)
    : ImapResult(true), uidvalidity(uidvalidity), uid(uid)
{
}

//-----------------------------------------------------------------------------

bool operator==(const ImapMessageVector& lhs, const ImapMessageVector& rhs)
{
    if (lhs.size() != rhs.size()) return false;

    return std::equal(
        lhs.begin(),
        lhs.end(),
        rhs.begin(),
        [](const ImapMessageVector::value_type& msg1, const ImapMessageVector::value_type& msg2) {
            return msg1.uid == msg2.uid && msg1.flags == msg2.flags;
        });
}

} // namespace ymod_imap_client
