#ifndef _YMOD_IMAP_CLIENT_IMAP_FILTER_H_
#define _YMOD_IMAP_CLIENT_IMAP_FILTER_H_

#include <imap_protocol/preparser.h>

#include <yplatform/net/line_filter.h>
#include <boost/bind.hpp>
#include <utility>

using namespace std::string_literals;

namespace ymod_imap_client {

enum class ImapFilterState
{
    EXPECT_GREETING,
    EXPECT_COMMAND_CONTINUATION,
    EXPECT_RESPONSE_DONE,
};

class ImapFilter : public yplatform::net::proto_filter
{
    using StreamBuffer = boost::asio::basic_streambuf<std::allocator<char>>;
    using Iterator = boost::asio::buffers_iterator<StreamBuffer::const_buffers_type>;
    using Preparser = imap_protocol::Preparser<Iterator>;
    using PreparserPtr = std::shared_ptr<Preparser>;

public:
    std::pair<Iterator, bool> operator()(Iterator begin, Iterator end)
    {
        auto current = begin;
        while (current < end)
        {
            auto response = boost::make_iterator_range(current, end);
            auto res = (*preparser)(response);
            if (!res.commandComplete)
            {
                return makeResponse(current, false);
            }

            resetPreparser();
            if (*currentState == ImapFilterState::EXPECT_GREETING)
            {
                return makeResponse(current + res.totalSize, true);
            }

            if (*currentState == ImapFilterState::EXPECT_COMMAND_CONTINUATION &&
                res.commandTag == "+")
            {
                return makeResponse(current + res.totalSize, true);
            }

            if (res.commandTag != "*")
            {
                return makeResponse(current + res.totalSize, true);
            }

            if (boost::algorithm::istarts_with(response, "* BYE"s))
            {
                return makeResponse(current + res.totalSize, true);
            }

            current += res.totalSize;
        }
        return makeResponse(current, false);
    }

    void setState(ImapFilterState newState)
    {
        *currentState = newState;
    }

    std::pair<Iterator, bool> makeResponse(Iterator consumedEnd, bool finished)
    {
        return std::make_pair(consumedEnd, finished);
    }

    void resetPreparser()
    {
        *preparser = Preparser();
    }

private:
    PreparserPtr preparser = std::make_shared<Preparser>();
    std::shared_ptr<ImapFilterState> currentState =
        std::make_shared<ImapFilterState>(ImapFilterState::EXPECT_RESPONSE_DONE);
};

} // namespace ymod_imap_client

namespace boost { namespace asio {

template <>
struct is_match_condition<ymod_imap_client::ImapFilter> : public boost::true_type
{
};

} // namespace asio
} // namespace boost

#endif //_YMOD_IMAP_CLIENT_IMAP_FILTER_H_
