#pragma once

#include <ymod_lease/types.h>
#include <ymod_lease/lock_manager.h>

namespace ylease {

using buckets_resources_map = std::map<std::string, std::set<std::string>>;

// Bucket manager evenly divides n buckets with resources between m consumers.
// Firstly acquire bucket then lock all bucket resources.
// Class takes on input
// - single thread io_service for internal synchronization
// - lock_manager to acquire buckets
// - lease_node to acquire resources
template <typename LeaseNode>
class bucket_manager
    : public std::enable_shared_from_this<bucket_manager<LeaseNode>>
    , public yplatform::log::contains_logger
{
    using lock_manager = lock_manager<LeaseNode>;

public:
    bucket_manager(
        boost::asio::io_service* io,
        const std::shared_ptr<lock_manager>& lock_manager,
        const std::shared_ptr<LeaseNode>& lease_node)
        : io_(io)
        , lock_manager_(lock_manager)
        , lease_node_(lease_node)
        , my_node_id_(lease_node->node_id())
    {
    }

    void init(const resource_cb& on_acquire_resource, const resource_cb& on_release_resource)
    {
        on_acquire_resource_ = on_acquire_resource;
        on_release_resource_ = on_release_resource;

        auto acquire_bucket_cb =
            io_->wrap(weak_bind(&bucket_manager::on_acquire_bucket, shared_from(this), ph::_1));
        auto release_bucket_cb =
            io_->wrap(weak_bind(&bucket_manager::on_release_bucket, shared_from(this), ph::_1));
        lock_manager_->init(acquire_bucket_cb, release_bucket_cb);
    }

    void add_buckets(const buckets_resources_map& buckets)
    {
        io_->post([this, self = shared_from(this), buckets] { add_buckets_impl(buckets); });
    }

    void del_buckets(const std::vector<std::string>& buckets)
    {
        io_->post([this, self = shared_from(this), buckets] {
            for (auto& bucket_name : buckets)
            {
                auto it = buckets_.find(bucket_name);
                if (it == buckets_.end())
                {
                    YLOG_L(info) << "try to delete not existing bucket " << bucket_name;
                    continue;
                }
                YLOG_L(info) << "delete bucket " << bucket_name;
                std::set<std::string> resources_to_del;
                auto& bucket = it->second;
                for (auto& [resource_name, resource] : bucket.resources)
                {
                    resources_to_del.insert(resource_name);
                }
                del_resources_from_bucket_impl(bucket_name, resources_to_del);
                buckets_.erase(it);
            }
            lock_manager_->on_del_resources(buckets);
        });
    }

    void upd_buckets(const buckets_resources_map& buckets)
    {
        io_->post([this, self = shared_from(this), buckets] {
            for (auto& bucket : buckets)
            {
                auto& bucket_name = bucket.first;
                auto bucket_it = buckets_.find(bucket_name);
                if (bucket_it == buckets_.end())
                {
                    YLOG_L(info) << "try to update not existing bucket " << bucket_name;
                    add_buckets_impl({ bucket });
                    continue;
                }
                YLOG_L(info) << "update bucket " << bucket_name;
                auto& resources = bucket_it->second.resources;
                auto& resources_to_upd = bucket.second;
                std::set<std::string> resources_to_add;
                for (auto& resource_to_upd : resources_to_upd)
                {
                    if (!resources.count(resource_to_upd))
                    {
                        resources_to_add.insert(resource_to_upd);
                    }
                }
                add_resources_to_bucket_impl(bucket_name, resources_to_add);

                std::set<std::string> resources_to_del;
                for (auto& resource : resources)
                {
                    auto& name = resource.first;
                    if (!resources_to_upd.count(name))
                    {
                        resources_to_del.insert(name);
                    }
                }
                del_resources_from_bucket_impl(bucket_name, resources_to_del);
            }
        });
    }

    void add_resources_to_bucket(
        const std::string& bucket_name,
        const std::set<std::string>& resources_to_add)
    {
        io_->post([this, self = shared_from(this), bucket_name, resources_to_add] {
            add_resources_to_bucket_impl(bucket_name, resources_to_add);
        });
    }

    void del_resources_from_bucket(
        const std::string& bucket_name,
        const std::set<std::string>& resources_to_del)
    {
        io_->post([this, self = shared_from(this), bucket_name, resources_to_del] {
            del_resources_from_bucket_impl(bucket_name, resources_to_del);
        });
    }

    void release_buckets_for(const std::vector<std::string>& buckets, const time_duration& duration)
    {
        io_->post([this, self = shared_from(this), buckets, duration] {
            lock_manager_->release_resources_for(buckets, duration);
        });
    }

    void set_resource_value(const std::string& resource, const std::string& value)
    {
        io_->post([this, self = shared_from(this), resource, value] {
            for (auto& bucket : buckets_)
            {
                if (!bucket.second.has_resource(resource))
                {
                    continue;
                }
                if (is_owner(bucket.second.resources[resource]))
                {
                    lease_node_->update_acquire_value(resource, value);
                }
                break;
            }
        });
    }

    void get_resource_value(
        const std::string& resource,
        const std::function<void(const std::string&)>& cb)
    {
        io_->post([this, self = shared_from(this), resource, cb] {
            for (auto& bucket : buckets_)
            {
                if (!bucket.second.has_resource(resource))
                {
                    continue;
                }
                return cb(bucket.second.resources[resource].value);
            }
            return cb(std::string());
        });
    }

    void check_is_owner(const std::string& resource, const std::function<void(bool)>& cb)
    {
        io_->post([this, self = shared_from(this), resource, cb] {
            for (auto& bucket : buckets_)
            {
                if (!bucket.second.has_resource(resource))
                {
                    continue;
                }
                return cb(is_owner(bucket.second.resources[resource]));
            }
            return cb(false);
        });
    }

    void get_acquired_buckets(const std::function<void(const std::vector<std::string>&)>& cb)
    {
        io_->post([this, self = shared_from(this), cb] {
            std::vector<std::string> acquired_buckets;
            for (auto& it : buckets_)
            {
                if (!it.second.is_owner)
                {
                    continue;
                }
                acquired_buckets.push_back(it.first);
            }
            cb(acquired_buckets);
        });
    }

private:
    struct bucket
    {
        bool has_resource(const std::string& resource)
        {
            return resources.count(resource);
        }

        bool is_owner = false;
        resources_map resources;
    };

    void add_buckets_impl(const buckets_resources_map& buckets)
    {
        std::vector<std::string> buckets_to_add;
        for (auto& bucket : buckets)
        {
            auto& name = bucket.first;
            auto& resources_to_add = bucket.second;
            auto it = buckets_.find(name);
            if (it != buckets_.end())
            {
                YLOG_L(info) << "try to add existing bucket " << name;
                continue;
            }
            YLOG_L(info) << "add bucket " << name;
            buckets_[name];
            add_resources_to_bucket_impl(name, resources_to_add);
            buckets_to_add.push_back(name);
        }
        lock_manager_->on_add_resources(buckets_to_add);
    }

    void add_resources_to_bucket_impl(
        const std::string& bucket_name,
        const std::set<std::string>& resources_to_add)
    {
        auto it = buckets_.find(bucket_name);
        if (it == buckets_.end())
        {
            YLOG_L(info) << "try to add resources to not existing bucket " << bucket_name;
            add_buckets_impl({ { bucket_name, resources_to_add } });
            return;
        }
        YLOG_L(info) << "add resources to bucket " << bucket_name;
        auto& resources = it->second.resources;
        for (auto& resource_name : resources_to_add)
        {
            if (resources.count(resource_name))
            {
                continue;
            }
            bind_resource(resource_name);
            auto deleted_it = deleted_owned_resources_.find(resource_name);
            resource_info* resource = nullptr;
            if (deleted_it != deleted_owned_resources_.end())
            {
                resource = &(resources[resource_name] = deleted_it->second);
                deleted_owned_resources_.erase(deleted_it);
            }
            else
            {
                resource = &(resources[resource_name] = resource_info());
            }
            if (it->second.is_owner)
            {
                start_acquire_lease(resource_name, *resource);
            }
            else
            {
                start_read_only(resource_name, *resource);
            }
        }
    }

    void del_resources_from_bucket_impl(
        const std::string& bucket_name,
        const std::set<std::string>& resources_to_del)
    {
        auto bucket_it = buckets_.find(bucket_name);
        if (bucket_it == buckets_.end())
        {
            YLOG_L(info) << "try to del resources from not existing bucket " << bucket_name;
            return;
        }
        YLOG_L(info) << "del resources from bucket " << bucket_name;
        auto& resources = bucket_it->second.resources;
        for (auto& resource : resources_to_del)
        {
            auto resource_it = resources.find(resource);
            if (resource_it == resources.end())
            {
                continue;
            }
            auto& resource_info = resource_it->second;
            stop_acquire_lease(resource, resource_info);
            if (is_owner(resource_it->second))
            {
                deleted_owned_resources_.emplace(resource_it->first, resource_it->second);
            }
            resources.erase(resource_it);
        }
    }

    void on_acquire_bucket(const std::string& bucket)
    {
        auto it = buckets_.find(bucket);
        if (it == buckets_.end())
        {
            return;
        }
        it->second.is_owner = true;
        auto& resources = it->second.resources;
        for (auto& resource : resources)
        {
            start_acquire_lease(resource.first, resource.second);
        }
    }

    void on_release_bucket(const std::string& bucket)
    {
        auto it = buckets_.find(bucket);
        if (it == buckets_.end())
        {
            return;
        }
        it->second.is_owner = false;
        for (auto& resource : it->second.resources)
        {
            start_read_only(resource.first, resource.second);
        }
    }

    void on_busy(
        const std::string& name,
        const std::string& node_id,
        ballot_t,
        const std::string& value)
    {
        YLOG_L(info) << "resource \"" << name << "\" is busy; owner_id=\"" << node_id
                     << "\", value=\"" << value << "\"";
        auto [resource, deleted] = find_resource(name);
        if (resource)
        {
            bool was_owner = is_owner(*resource);
            resource->owner_id = node_id;
            resource->value = value;

            if (is_owner(*resource) && !was_owner)
            {
                io_->post(std::bind(on_acquire_resource_, name));
            }
            else if (!is_owner(*resource) && was_owner)
            {
                io_->post(std::bind(on_release_resource_, name));
                if (deleted)
                {
                    deleted_owned_resources_.erase(name);
                }
            }
        }
    }

    void on_free(const std::string& name)
    {
        YLOG_L(info) << "resource \"" << name << "\" is free";
        auto [resource, deleted] = find_resource(name);
        if (resource)
        {
            bool was_owner = is_owner(*resource);
            resource->owner_id.clear();
            resource->value.clear();
            if (was_owner)
            {
                io_->post(std::bind(on_release_resource_, name));
                if (deleted)
                {
                    deleted_owned_resources_.erase(name);
                }
            }
        }
    }

    bool is_owner(const resource_info& resource) const
    {
        return resource.owner_id == my_node_id_;
    }

    void bind_resource(const std::string& name)
    {
        YLOG_L(info) << "bind resource " << name;
        auto busy_cb = io_->wrap(
            weak_bind(&bucket_manager::on_busy, shared_from(this), ph::_1, ph::_2, ph::_3, ph::_4));
        auto free_cb = io_->wrap(weak_bind(&bucket_manager::on_free, shared_from(this), ph::_1));
        lease_node_->bind(name, busy_cb, free_cb);
    }

    void start_acquire_lease(const std::string& name, resource_info& resource)
    {
        YLOG_L(info) << "start_acquire_lease " << name;
        resource.is_acquiring = true;
        io_->post(std::bind(&LeaseNode::start_acquire_lease, lease_node_, name));
    }

    void start_read_only(const std::string& name, resource_info& resource)
    {
        YLOG_L(info) << "start_read_only " << name;
        io_->post(std::bind(&LeaseNode::start_read_only, lease_node_, name));
        resource.is_acquiring = false;
    }

    void stop_acquire_lease(const std::string& name, resource_info& resource)
    {
        YLOG_L(info) << "stop_acquire_lease " << name;
        io_->post(std::bind(&LeaseNode::stop_acquire_lease, lease_node_, name));
        resource.is_acquiring = false;
    }

    std::pair<resource_info*, bool> find_resource(const std::string& resource_name)
    {
        for (auto& [bucket_name, bucket] : buckets_)
        {
            auto it = bucket.resources.find(resource_name);
            if (it != bucket.resources.end())
            {
                return { &it->second, false };
            }
        }
        auto it = deleted_owned_resources_.find(resource_name);
        if (it != deleted_owned_resources_.end())
        {
            return { &it->second, true };
        }
        return { nullptr, false };
    }

    boost::asio::io_service* io_;
    std::shared_ptr<lock_manager> lock_manager_;
    std::shared_ptr<LeaseNode> lease_node_;
    std::string my_node_id_;
    resource_cb on_acquire_resource_;
    resource_cb on_release_resource_;
    std::map<std::string, bucket> buckets_;
    resources_map
        deleted_owned_resources_; // used for call on_release_resource, cleared after doing this
};

}
