#pragma once

#include <ymod_lease/types.h>
#include <yplatform/log.h>

#include <boost/asio.hpp>

#include <map>
#include <memory>
#include <functional>

namespace ylease {

namespace ph = std::placeholders;

struct resource_info
{
    std::string owner_id;
    std::string value;
    bool is_acquiring = false;
    time_point released_ts = {};
};
using resources_map = std::map<std::string, resource_info>;
using resource_cb = std::function<void(const std::string&)>;
using resource_value_cb = std::function<void(const std::string&, const std::string&)>;
using resources_cb = std::function<void(const std::vector<std::string>&)>;
using value_cb = resource_cb;

template <typename LeaseNode>
class lock_manager
    : public std::enable_shared_from_this<lock_manager<LeaseNode>>
    , public yplatform::log::contains_logger
{
public:
    using ballot_t = int64_t;

    lock_manager(
        boost::asio::io_service* io,
        const std::shared_ptr<LeaseNode>& lease_node,
        const std::size_t max_owned_count,
        const std::size_t extra_acquire_count)
        : io_(io)
        , lease_node_(lease_node)
        , max_owned_count_(max_owned_count)
        , extra_acquire_count_(extra_acquire_count)
        , my_node_id_(lease_node->node_id())
    {
    }

    void init(const resource_cb& on_acquire_resource, const resource_cb& on_release_resource)
    {
        on_acquire_resource_ = on_acquire_resource;
        on_release_resource_ = on_release_resource;

        auto upd_peers_count_cb =
            weak_bind(&lock_manager::update_peers_count, shared_from(this), ph::_1);
        lease_node_->subscribe_peers_count(upd_peers_count_cb);
    }

    void subscribe_for_value_update(const resource_value_cb& on_resource_value_updated)
    {
        on_resource_value_updated_ = on_resource_value_updated;
    }

    void on_add_resources(const std::vector<std::string>& resources)
    {
        io_->post([this, self = shared_from(this), resources] {
            auto busy_cb =
                io_->wrap(weak_bind(&lock_manager::on_busy, self, ph::_1, ph::_2, ph::_3, ph::_4));
            auto free_cb = io_->wrap(weak_bind(&lock_manager::on_free, self, ph::_1));
            for (auto& resource : resources)
            {
                if (resources_.find(resource) != resources_.end())
                {
                    YLOG_L(warning) << "resource already exist" << resource;
                    continue;
                }
                YLOG_L(info) << "add resource " << resource;
                auto it = deleted_owned_resources_.find(resource);
                if (it != deleted_owned_resources_.end())
                {
                    resources_[resource] = it->second;
                    deleted_owned_resources_.erase(it);
                }
                else
                {
                    resources_[resource] = resource_info();
                }
                lease_node_->bind(resource, busy_cb, free_cb);
                if (owned_count() < max_owned_count() && acquiring_count() < max_acquiring_count())
                {
                    start_acquire_lease(resource);
                }
                else
                {
                    start_read_only(resource);
                }
            }
        });
        io_->post(std::bind(&lock_manager::update_locks, shared_from(this)));
    }

    void on_del_resources(const std::vector<std::string>& resources)
    {
        io_->post([this, self = shared_from(this), resources] {
            for (auto& name : resources)
            {
                auto it = resources_.find(name);
                if (it == resources_.end())
                {
                    YLOG_L(warning) << "delete not existing resource " << name;
                    continue;
                }
                YLOG_L(info) << "del resource " << name;
                stop_acquire_lease(name);
                if (is_owner(it->second))
                {
                    deleted_owned_resources_.emplace(name, it->second);
                }
                resources_.erase(it);
            }
        });
        io_->post(std::bind(&lock_manager::update_locks, shared_from(this)));
    }

    void update_peers_count(std::size_t count)
    {
        if (!count) count = 1;
        io_->post([this, self = shared_from(this), count] {
            YLOG_L(info) << "update peers count " << count;
            peers_count_ = count;
            update_locks();
        });
    }

    void get_resource_value(
        const std::string& resource,
        const std::function<void(const std::string&)>& cb)
    {
        io_->post([this, self = shared_from(this), resource, cb] {
            auto it = resources_.find(resource);
            if (it != resources_.end())
            {
                auto& resource = it->second;
                cb(resource.value);
            }
            else
            {
                cb(std::string());
            }
        });
    }

    void check_is_owner(const std::string& resource, const std::function<void(bool)>& cb)
    {
        io_->post([this, self = shared_from(this), resource, cb] { cb(is_owner(resource)); });
    }

    void set_resource_value(const std::string& name, const std::string& value)
    {
        io_->post([this, self = shared_from(this), name, value] {
            auto it = resources_.find(name);
            if (it == resources_.end())
            {
                YLOG_L(error) << "set value for not existing resource, resource=" << name
                              << " value=" << value;
                return;
            }
            auto& resource = it->second;
            if (is_owner(resource))
            {
                lease_node_->update_acquire_value(name, value);
            }
        });
    }

    void release_resources_for(
        const std::vector<std::string>& resources,
        const time_duration& duration)
    {
        time_point released_ts = clock::now() + duration;
        io_->post([this, self = shared_from(this), resources, released_ts] {
            for (auto& name : resources)
            {
                auto it = resources_.find(name);
                if (it != resources_.end())
                {
                    it->second.released_ts = released_ts;
                }
            }
            auto timer = std::make_shared<steady_timer>(*io_, released_ts);
            timer->async_wait([this, self, timer, resources](error_code err) {
                if (err)
                {
                    // just log
                    YLOG_L(error) << "release resources error: " << err.message();
                }
                io_->post(std::bind(&lock_manager::update_locks, self));
            });
            io_->post(std::bind(&lock_manager::update_locks, self));
        });
    }

    void get_acquired_resources(const resources_cb& cb)
    {
        io_->post([this, self = shared_from(this), cb] {
            std::vector<std::string> ret;
            for (auto& [name, resource] : resources_)
            {
                if (is_owner(resource))
                {
                    ret.emplace_back(name);
                }
            }
            cb(ret);
        });
    }

private:
    void on_busy(
        const std::string& name,
        const std::string& node_id,
        ballot_t,
        const std::string& value)
    {
        YLOG_L(info) << "resource \"" << name << "\" is busy; owner_id=\"" << node_id
                     << "\", value=\"" << value << "\"";
        auto [resource, deleted] = find_resource(name);
        if (!resource)
        {
            return;
        }
        bool was_owner = is_owner(*resource);
        resource->owner_id = node_id;
        resource->value = value;
        if (is_owner(*resource) && !was_owner)
        {
            ++owned_count_;
            io_->post(std::bind(on_acquire_resource_, name));
        }
        else if (!is_owner(*resource) && was_owner)
        {
            --owned_count_;
            io_->post(std::bind(on_release_resource_, name));
            if (deleted)
            {
                deleted_owned_resources_.erase(name);
            }
        }
        io_->post(std::bind(&lock_manager::update_locks, shared_from(this)));
        if (on_resource_value_updated_)
        {
            io_->post(std::bind(on_resource_value_updated_, name, value));
        }
    }

    void on_free(const std::string& name)
    {
        YLOG_L(info) << "resource \"" << name << "\" is free";
        auto [resource, deleted] = find_resource(name);
        if (!resource)
        {
            return;
        }
        bool was_owner = is_owner(*resource);
        resource->owner_id = "";
        resource->value = "";
        if (was_owner)
        {
            --owned_count_;
            io_->post(std::bind(on_release_resource_, name));
            if (deleted)
            {
                deleted_owned_resources_.erase(name);
            }
        }
        io_->post(std::bind(&lock_manager::update_locks, shared_from(this)));
    }

    void update_locks()
    {
        YLOG_L(info) << "update_locks owned " << owned_count() << " acquiring " << acquiring_count()
                     << " max_owned " << max_owned_count() << " max_acquiring "
                     << max_acquiring_count();
        start_read_only_for_released_resources();
        if (owned_count() >= max_owned_count())
        {
            start_read_only_for_not_owned();
            start_read_only_for_extra_owned();
        }
        else
        {
            start_read_only_for_owned_by_other_nodes();
            if (acquiring_count() >= max_acquiring_count())
            {
                start_read_only_for_extra_acquiring();
            }
            else
            {
                start_acquire_lease_for_free_resources();
            }
        }
    }

    void start_read_only_for_not_owned()
    {
        for (auto& pair : resources_)
        {
            auto& name = pair.first;
            auto& resource = pair.second;
            if (resource.is_acquiring && !is_owner(resource))
            {
                start_read_only(name);
            }
        }
    }

    void start_read_only_for_extra_owned()
    {
        std::size_t owned_and_acquiring_count = 0;
        for (auto& [name, resource] : resources_)
        {
            if (is_owner(resource) && resource.is_acquiring)
            {
                if (owned_and_acquiring_count < max_owned_count())
                {
                    ++owned_and_acquiring_count;
                }
                else
                {
                    start_read_only(name);
                }
            }
        }
    }

    void start_read_only_for_extra_acquiring()
    {
        for (auto& pair : resources_)
        {
            if (acquiring_count() <= max_acquiring_count()) break;
            auto& name = pair.first;
            auto& resource = pair.second;
            if (resource.is_acquiring && !is_owner(resource))
            {
                start_read_only(name);
            }
        }
    }

    std::vector<std::string> get_resources_in_random_order() const
    {
        std::vector<std::string> ret;
        ret.reserve(resources_.size());
        for (auto& resource : resources_)
        {
            ret.push_back(resource.first);
        }
        std::random_shuffle(ret.begin(), ret.end());
        return ret;
    }

    void start_acquire_lease_for_free_resources()
    {
        auto resources = get_resources_in_random_order();
        for (auto& name : resources)
        {
            if (acquiring_count() >= max_acquiring_count()) break;
            auto& resource = resources_[name];
            if (clock::now() < resource.released_ts) continue;
            if (!resource.is_acquiring && resource.owner_id.empty())
            {
                start_acquire_lease(name);
            }
        }
    }

    void start_read_only_for_owned_by_other_nodes()
    {
        for (auto& pair : resources_)
        {
            auto& name = pair.first;
            auto& resource = pair.second;
            if (resource.is_acquiring && resource.owner_id.size() && !is_owner(resource))
            {
                start_read_only(name);
            }
        }
    }

    void start_read_only_for_released_resources()
    {
        for (auto& pair : resources_)
        {
            auto& name = pair.first;
            auto& resource = pair.second;
            if (resource.is_acquiring && clock::now() < resource.released_ts)
            {
                start_read_only(name);
            }
        }
    }

    void start_read_only(const std::string& name)
    {
        YLOG_L(info) << "start_read_only " << name;
        auto& resource = resources_[name];
        if (resource.is_acquiring)
        {
            --acquiring_count_;
            resource.is_acquiring = false;
        }
        io_->post(std::bind(&LeaseNode::start_read_only, lease_node_, name));
    }

    void start_acquire_lease(const std::string& name)
    {
        YLOG_L(info) << "start_acquire_lease " << name;
        auto& resource = resources_[name];
        if (!resource.is_acquiring)
        {
            ++acquiring_count_;
            resource.is_acquiring = true;
        }
        io_->post(std::bind(&LeaseNode::start_acquire_lease, lease_node_, name));
    }

    void stop_acquire_lease(const std::string& name)
    {
        YLOG_L(info) << "stop_acquire_lease " << name;
        auto& resource = resources_[name];
        if (resource.is_acquiring)
        {
            --acquiring_count_;
            resource.is_acquiring = false;
        }
        io_->post(std::bind(&LeaseNode::stop_acquire_lease, lease_node_, name));
    }

    std::pair<resource_info*, bool> find_resource(const std::string& name)
    {
        auto it = resources_.find(name);
        if (it != resources_.end())
        {
            return { &it->second, false };
        }
        it = deleted_owned_resources_.find(name);
        if (it != deleted_owned_resources_.end())
        {
            return { &it->second, true };
        }
        return { nullptr, false };
    }

    std::size_t owned_count() const
    {
        return owned_count_;
    }

    std::size_t acquiring_count() const
    {
        return acquiring_count_;
    }

    std::size_t max_owned_count() const
    {
        std::size_t preffered_count = (resources_.size() + peers_count_ - 1u) / peers_count_;
        return std::min(preffered_count, max_owned_count_);
    }

    std::size_t max_acquiring_count() const
    {
        return max_owned_count() + extra_acquire_count_;
    }

    bool is_owner(const std::string& name)
    {
        return is_owner(resources_[name]);
    }

    bool is_owner(const resource_info& resource) const
    {
        return resource.owner_id == my_node_id_;
    }

    boost::asio::io_service* io_;
    std::shared_ptr<LeaseNode> lease_node_;
    std::size_t max_owned_count_;
    std::size_t extra_acquire_count_;
    std::size_t peers_count_ = 1u;
    std::size_t owned_count_ = 0u;
    std::size_t acquiring_count_ = 0u;
    std::string my_node_id_;
    resource_cb on_acquire_resource_;
    resource_cb on_release_resource_;
    resource_value_cb on_resource_value_updated_;
    resources_map resources_;
    resources_map
        deleted_owned_resources_; // used for call on_release_resource, cleared after doing this
};

}
