#include "arbiter_impl.h"

#include <msgpack.hpp>
#include <sstream>
#include <sintimers/queue.h>
#include "time_millis.h"

#define TIMEOUTS_LOOP_CHECK_INTERVAL 100

namespace ylease {

using std::string;

namespace {
void removeDuplicates(address_list& list)
{
    std::sort(list.begin(), list.end());
    list.erase(std::unique(list.begin(), list.end()), list.end());
}
}

void arbiter_impl::init(const yplatform::ptree& xml)
{
    auto lease_log_id = xml.get<std::string>("lease_log_id", "global");
    auto custom_logger = yplatform::log::source(YGLOBAL_LOG_SERVICE, lease_log_id);
    custom_logger.set_log_prefix(this->name());
    this->logger(custom_logger);
    yplatform_log_wrapper_.logger(custom_logger);

    string netch_module_name = xml.get("netch_module", "netch");
    netch_ = yplatform::find<netch>(netch_module_name);
    netch_message_type netch_base_type =
        xml.get<netch_message_type>("netch_message_base_type", 9000);
    netch_codes_.reset(new message_types(ymod_messenger::message_type_USER + netch_base_type));

    // init arbiter as netch node
    arbiter_id_ = uuids_generator_().substr(0, 7);
    arbiter_id_ = netch_->my_address() + "-" + arbiter_id_;

    auto reactor = yplatform::global_net_reactor;
    timers_queue_ = boost::make_shared<timers::queue>(reactor);

    if (xml.get("count_nodes", false))
    {
        nodes_registry_ = std::make_shared<nodes_registry>(
            *reactor->io(),
            [this, weak_self = weak_from(this)](const string& tag, address_list list) {
                if (auto self = weak_self.lock())
                {
                    nodes_count_msg msg = { tag, list.size() };
                    removeDuplicates(list);
                    for (auto& address : list)
                    {
                        send_nodes_count(address, msg);
                    }
                }
            });

        node_inactive_timeout_ = xml.get<time_duration>("node_inactive_timeout");
    }

    // bind to events and messages
    subscribe_mcast_messages();

    ARBITER_LOG(info) << "inited " << arbiter_id_;

    reload(xml);
}

void arbiter_impl::reload(const yplatform::ptree& xml)
{
    verbose_logging_ = xml.get<bool>("verbose_logging", 0);
    yplatform_log_wrapper_.set_enabled(verbose_logging_);
    ARBITER_LOG(info) << "reloaded";
}

void arbiter_impl::start()
{
    stop_ = false;
}

void arbiter_impl::fini()
{
    {
        scoped_lock lock(mux_);
        stop_ = true;
    }
    resource_servers::iterator it = resource_servers_.begin();
    for (; it != resource_servers_.end(); ++it)
    {
        it->second->fini();
    }
    resource_servers_.clear();
    timers_queue_.reset();
}

void arbiter_impl::subscribe_mcast_messages()
{
    netch_->bind_messages<prepare_msg>(
        boost::bind(&arbiter_impl::on_prepare_message, this, _1, _2), netch_codes_->prepare);
    netch_->bind_messages<accept_msg>(
        boost::bind(&arbiter_impl::on_accept_message, this, _1, _2), netch_codes_->accept);
}

void arbiter_impl::on_prepare_message(const netch_address& sender, prepare_msg msg)
{
    try
    {
        get_resource_server(msg.resource_name)
            ->prepare(
                msg,
                boost::bind(&arbiter_impl::send_promise, this, sender, _1),
                boost::bind(&arbiter_impl::send_reject, this, sender, _1));

        // ignore nodes with no tag - otherwise
        // arbiters will send too many unnecessary messages
        if (nodes_registry_ && msg.tag.size())
        {
            nodes_registry_->insert(
                sender,
                msg.node,
                msg.tag,
                milliseconds(msg.vote_interval) + node_inactive_timeout_);
        }
    }
    catch (std::exception& e)
    {
        ARBITER_RESLOG(error, msg.resource_name)
            << "failed to process prepare msg: " + string(e.what());
        return;
    }
}

void arbiter_impl::on_accept_message(const netch_address& sender, accept_msg msg)
{
    try
    {
        get_resource_server(msg.resource_name)
            ->accept(
                msg,
                boost::bind(&arbiter_impl::on_accepted, this, sender, _1),
                boost::bind(&arbiter_impl::send_reject, this, sender, _1));
    }
    catch (std::exception& e)
    {
        ARBITER_RESLOG(error, msg.resource_name)
            << "failed to process accept msg: " + string(e.what());
        return;
    }
}

void arbiter_impl::send_promise(const netch_address& to, const promise_msg& message)
{
    if (verbose_logging_)
        ARBITER_LOG(debug) << "sending promise for ballot:" << message.ballot << " to " << to;
    netch_->send(to, message, netch_codes_->promise);
}

void arbiter_impl::send_reject(const netch_address& to, const reject_msg& message)
{
    if (verbose_logging_)
        ARBITER_LOG(debug) << "sending REJECT for ballot:" << message.ballot << " to " << to;
    netch_->send(to, message, netch_codes_->reject);
}

void arbiter_impl::send_accepted(const netch_address& to, const accepted_msg& message)
{
    if (verbose_logging_) ARBITER_LOG(debug) << "sending accepted for ballot:" << message.ballot;
    netch_->send(to, message, netch_codes_->accepted);
}

void arbiter_impl::send_nodes_count(const netch_address& to, const nodes_count_msg& message)
{
    if (verbose_logging_)
        ARBITER_LOG(info) << "sending nodes count " << message.count << " for tag:" << message.tag
                          << " to " << to;
    netch_->send(to, message, netch_codes_->nodes_count);
}

arbiter_impl::resource_server_ptr arbiter_impl::get_resource_server(const string& resource_name)
{
    if (resource_name.empty())
    {
        throw std::domain_error("[arbiter_impl::get_resource_server] resource name can't be empty");
    }
    scoped_lock lock(mux_);
    if (stop_)
    {
        throw std::domain_error("[arbiter_impl::get_resource_server] module stopped");
    }
    resource_servers::iterator found = resource_servers_.find(resource_name);
    if (found == resource_servers_.end())
    {
        resource_server_ptr newServer(new resource_server(
            arbiter_id_, resource_name, *timers_queue_, yplatform_log_wrapper_));
        /// @todo check result
        return resource_servers_.insert(found, std::make_pair(resource_name, newServer))->second;
    }
    return found->second;
}

void arbiter_impl::on_accepted(const netch_address& sender, const accepted_msg& message)
{
    if (verbose_logging_)
        ARBITER_RESLOG(debug, message.resource_name)
            << "on_accepted: ACCEPTED on ballot " << message.ballot;
    send_accepted(sender, message);
}

}
