#include <ymod_lease/node.h>

#include "node_impl.h"

namespace ylease {

namespace {
settings parse_settings(const yplatform::ptree& ptree)
{
    settings st;
    st.parse_ptree(ptree);
    return st;
}

auto make_reactor(boost::asio::io_service& io)
{
    return boost::make_shared<yplatform::reactor>(std::make_shared<yplatform::io_pool>(io, 1));
}
}

node::node(yplatform::reactor& reactor, const yplatform::ptree& ptree)
    : node(reactor, parse_settings(ptree))
{
}

node::node(boost::asio::io_service& io, const yplatform::ptree& ptree)
    : node(*make_reactor(io), parse_settings(ptree))
{
}

node::node(boost::asio::io_service& io, const settings& st) : node(*make_reactor(io), st)
{
}

node::node(yplatform::reactor& reactor, const settings& st) : impl_(std::make_shared<node::impl>())
{
    impl_->logger(*reactor.io());
    impl_->init(reactor, st);
}

void node::reload(const settings& st)
{
    impl_->reload(st);
}

void node::reload(const yplatform::ptree& ptree)
{
    reload(parse_settings(ptree));
}

void node::fini()
{
    impl_->fini();
}

void node::set_verbose_logging(bool verbose)
{
    impl_->set_verbose_logging(verbose);
}

void node::logger(const yplatform::log::source& logger)
{
    yplatform::module::logger(logger);
    impl_->logger(logger);
}

const node_id& node::node_id()
{
    return impl_->node_id();
}

void node::bind(
    const std::string& resource_name,
    const busy_callback& busy_callback,
    const free_callback& free_callback)
{
    impl_->bind(resource_name, busy_callback, free_callback);
}

void node::subscribe_peers_count(const peers_count_callback& callback)
{
    impl_->subscribe_peers_count(callback);
}

void node::start_acquire_lease(const std::string& resource_name)
{
    impl_->start_acquire_lease(resource_name);
}
void node::start_read_only(const std::string& resource_name)
{
    impl_->start_read_only(resource_name);
}

void node::update_acquire_value(const std::string& resource_name, const value& value)
{
    impl_->update_acquire_value(resource_name, value);
}

void node::update_lease_time(const std::string& resource_name, const time_duration& time)
{
    impl_->update_lease_time(resource_name, time);
}

void node::stop_acquire_lease(const std::string& resource_name)
{
    impl_->stop_acquire_lease(resource_name);
}

std::string node::address_from_id(const ylease::node_id& node_id)
{
    return impl_->address_from_id(node_id);
}

}
