#include "node_impl.h"

#include <sstream>
#include <sintimers/queue.h>
#include "time_millis.h"
#include <yplatform/util/safe_call.h>

namespace ylease {

using std::string;

#define TIMEOUTS_LOOP_CHECK_INTERVAL 100

void node::impl::init(yplatform::reactor& reactor, const settings& st)
{
    io_ = reactor.io();
    st_ = st;

    if (st_.lease_log_id.size())
    {
        yplatform_log_wrapper_.logger(yplatform::log::source(*io_, st_.lease_log_id));
    }
    else
    {
        yplatform_log_wrapper_.logger(logger());
    }

    netch_ = yplatform::find<netch>(*io_, st.netch_module_name);
    netch_codes_.reset(new message_types(ymod_messenger::message_type_USER + st.netch_base_type));

    if (st_.acquire_lease_timeout > st_.max_lease_time)
    {
        throw std::domain_error(
            "bad lease timeout settings: acquire_lease_timeout > max_lease_time");
    }

    timers_queue_ =
        boost::make_shared<timers::queue>(yplatform::reactor::make_not_owning_copy(reactor));

    strand_.reset(new boost::asio::io_service::strand(*io_));

    node_id_ = uuids_generator_().substr(0, 7);
    node_id_ = netch_->my_address() + "-" + node_id_;
    NODE_LOG(info) << "lease node id: " << node_id_;

    // bind to events and messages
    subscribe_mcast_messages();

    set_verbose_logging(st_.verbose_logging);
}

void node::impl::reload(const settings& st)
{
    set_verbose_logging(st.verbose_logging);
}

void node::impl::fini()
{
    {
        scoped_lock lock(mux_);
        stop_ = true;
    }
    resource_clients::iterator it = resource_clients_.begin();
    for (; it != resource_clients_.end(); ++it)
    {
        it->second->fini();
    }
    resource_clients_.clear();
    timers_queue_.reset();
}

bool node::impl::stopped()
{
    scoped_lock lock(mux_);
    return stop_;
}

void node::impl::set_verbose_logging(bool value)
{
    yplatform_log_wrapper_.set_enabled(value);
    verbose_logging_ = value;
}

void node::impl::bind(
    const string& resource_name,
    const busy_callback& busy_callback,
    const free_callback& free_callback)
{
    if (resource_name.empty())
    {
        throw std::domain_error("[node::impl::bind] resource name can't be empty");
    }
    callbacks_.add(resource_name, busy_callback, free_callback);
}

void node::impl::subscribe_peers_count(const peers_count_callback& callback)
{
    peers_count_callbacks_.push_back(callback);
}

void node::impl::start_acquire_lease(const std::string& resource_name)
{
    if (stopped())
    {
        throw std::domain_error("[node::impl::start_acquire_lease] module has been stopped");
    }

    if (resource_name.empty())
    {
        throw std::domain_error("[node::impl::start_acquire_lease] resource name can't be empty");
    }
    get_resource_client(resource_name)->start_acquire_lease();
}

void node::impl::start_read_only(const string& resource_name)
{
    get_resource_client(resource_name)->start_read_only();
}

void node::impl::update_acquire_value(const std::string& resource_name, const value& value)
{
    if (stopped())
    {
        throw std::domain_error("[node::impl::update_acquire_value] module has been stopped");
    }

    if (resource_name.empty())
    {
        throw std::domain_error("[node::impl::update_acquire_value] resource name can't be empty");
    }
    get_resource_client(resource_name)->update_acquire_value(value);
}

void node::impl::update_lease_time(const std::string& resource_name, const time_duration& time)
{
    if (stopped())
    {
        throw std::domain_error("[node::impl::update_lease_time] module has been stopped");
    }

    if (resource_name.empty())
    {
        throw std::domain_error("[node::impl::update_lease_time] resource name can't be empty");
    }
    get_resource_client(resource_name)->update_lease_time(time);
}

void node::impl::stop_acquire_lease(const string& resource_name)
{
    if (stopped())
    {
        throw std::domain_error("[node::impl::stop_acquire_lease] module has been stopped");
    }

    if (resource_name.empty())
    {
        throw std::domain_error("[node::impl::stop_acquire_lease] resource name can't be empty");
    }
    get_resource_client(resource_name)->stop_acquire_lease();
}

void node::impl::subscribe_mcast_messages()
{
    netch_->bind_messages<promise_msg>(
        boost::bind(&node::impl::on_promise_message, this, _1, _2), netch_codes_->promise);
    netch_->bind_messages<reject_msg>(
        boost::bind(&node::impl::on_reject_message, this, _1, _2), netch_codes_->reject);
    netch_->bind_messages<accepted_msg>(
        boost::bind(&node::impl::on_accepted_message, this, _1, _2), netch_codes_->accepted);
    netch_->bind_messages<nodes_count_msg>(
        boost::bind(&node::impl::on_nodes_count_message, this, _1, _2), netch_codes_->nodes_count);
}

void node::impl::on_promise_message(const netch_address& /*sender*/, promise_msg msg)
{
    try
    {
        get_resource_client(msg.resource_name)->promise(msg);
    }
    catch (std::exception& e)
    {
        NODE_RESLOG(error, msg.resource_name)
            << "failed to process promise msg: " + string(e.what());
        return;
    }
}

void node::impl::on_reject_message(const netch_address& /*sender*/, reject_msg msg)
{
    try
    {
        get_resource_client(msg.resource_name)->reject(msg);
    }
    catch (std::exception& e)
    {
        NODE_RESLOG(error, msg.resource_name)
            << "failed to process reject msg: " + string(e.what());
        return;
    }
}

void node::impl::on_accepted_message(const netch_address& /*sender*/, accepted_msg msg)
{
    try
    {
        get_resource_client(msg.resource_name)->win(msg);
    }
    catch (std::exception& e)
    {
        NODE_RESLOG(error, msg.resource_name)
            << "failed to process accepted msg: " + string(e.what());
        return;
    }
}

void node::impl::on_nodes_count_message(const netch_address& sender, nodes_count_msg msg)
{
    if (verbose_logging_)
        NODE_LOG(info) << "received nodes count " << msg.count << " for tag " << msg.tag << " from "
                       << sender;

    if (st_.tag != msg.tag) return;
    if (peers_count_callbacks_.empty()) return;

    strand_->post([this, self = shared_from(this), sender, msg]() {
        if (peers_count_accumulator_.add(sender, msg.count))
        {
            for (auto& cb : peers_count_callbacks_)
            {
                yplatform::safe_call(cb, peers_count_accumulator_.value());
            }
        }
    });
}

void node::impl::broadcast_prepare(const prepare_msg& message)
{
    if (verbose_logging_)
        NODE_RESLOG(debug, message.resource_name)
            << "sending prepare for ballot:" << message.ballot;
    netch_->send_all(message, netch_codes_->prepare);
}

void node::impl::broadcast_accept(const accept_msg& message)
{
    if (verbose_logging_)
        NODE_RESLOG(debug, message.resource_name) << "sending accept for ballot:" << message.ballot;
    netch_->send_all(message, netch_codes_->accept);
}

const string& node::impl::node_id()
{
    return node_id_;
}

string node::impl::address_from_id(const ylease::node_id& id)
{
    auto colon_pos = id.find(':');
    return colon_pos != string::npos ? id.substr(0, colon_pos) : "";
}

node::impl::resource_client_ptr node::impl::get_resource_client(const string& resource_name)
{
    if (resource_name.empty())
    {
        throw std::domain_error("[node::impl::get_resource_client] resource name can't be empty");
    }
    scoped_lock lock(mux_);
    if (stop_)
    {
        throw std::domain_error("[node::impl::get_resource_client] module stopped");
    }
    resource_clients::iterator found = resource_clients_.find(resource_name);
    if (found == resource_clients_.end())
    {
        resource_client_ptr new_client(new resource_client(
            node_id(),
            resource_name,
            st_.tag,
            st_.acquire_lease_timeout,
            st_.max_lease_time,
            st_.arbiters_count,
            *timers_queue_,
            boost::bind(&node::impl::broadcast_prepare, this, _1),
            boost::bind(&node::impl::broadcast_accept, this, _1),
            boost::bind(&node::impl::on_win, this, resource_name, _1, _2),
            boost::bind(&node::impl::on_lose, this, resource_name),
            yplatform_log_wrapper_));
        /// @todo check result
        return resource_clients_.insert(found, std::make_pair(resource_name, new_client))->second;
    }
    return found->second;
}

void node::impl::on_win(const string& resource_name, const lease& lease, const ballot_t ballot)
{
    if (stopped())
    {
        return;
    }
    if (verbose_logging_)
    {
        NODE_RESLOG(debug, resource_name) << "BUSY resource for " << lease.duration;
    }
    strand_->post(boost::bind(
        &lease_callbacks_repository::call_busy_callbacks,
        &callbacks_,
        resource_name,
        ballot,
        lease));
}

void node::impl::on_lose(const string& resource_name)
{
    if (stopped())
    {
        return;
    }
    if (verbose_logging_) NODE_RESLOG(debug, resource_name) << "FREE resource";
    strand_->post(
        boost::bind(&lease_callbacks_repository::call_free_callbacks, &callbacks_, resource_name));
}

}
