/**
 * @file:   node.h
 * @author: nordsturm
 * © Yandex LLC.
 */

#ifndef YMOD_LEASE_NODE_IMPL_H
#define YMOD_LEASE_NODE_IMPL_H

#include <yplatform/module.h>
#include <yplatform/module_registration.h>
#include <yplatform/util/uuid_generator.h>
#include <ymod_messenger/ymod_messenger.h>

#include <ymod_lease/node.h>

#include "messages.h"
#include "network.h"
#include "types_priv.h"

#include "lease_config.h"
#include "time_millis.h"
#include "resource_client.h"
#include "timers_queue.h"
#include "logger.h"
#include "callbacks_repository.h"
#include "peers_count_accumulator.h"

namespace ylease {

class node::impl
    : public yplatform::log::contains_logger
    , public std::enable_shared_from_this<node::impl>
{
public:
    void init(yplatform::reactor& reactor, const settings& st);
    void reload(const settings& st);
    void fini();

    void set_verbose_logging(bool value);
    bool stopped();

    void bind(
        const string& resource_name,
        const busy_callback& busy_callback,
        const free_callback& free_callback);
    void subscribe_peers_count(const peers_count_callback& callback);
    void start_acquire_lease(const string& resource_name);
    void start_read_only(const string& resource_name);
    void update_acquire_value(const string& resource_name, const value& value);
    void update_lease_time(const std::string& resource_name, const time_duration& time);
    void stop_acquire_lease(const string& resource_name);

    const ylease::node_id& node_id();
    std::string address_from_id(const ylease::node_id&);

private:
    void subscribe_mcast_messages();

    void on_promise_message(const netch_address& sender, promise_msg msg);
    void on_reject_message(const netch_address& sender, reject_msg msg);
    void on_accepted_message(const netch_address& sender, accepted_msg msg);
    void on_nodes_count_message(const netch_address& sender, nodes_count_msg msg);

    void broadcast_prepare(const prepare_msg& message);
    void broadcast_accept(const accept_msg& message);

private:
    typedef boost::shared_ptr<resource_client> resource_client_ptr;
    typedef std::map<string, resource_client_ptr> resource_clients;

    resource_client_ptr get_resource_client(const string& resource_name);

    void on_win(const string& resource_name, const lease& lease, const ballot_t ballot);
    void on_lose(const string& resource_name);

    mutex mux_;

    boost::shared_ptr<netch> netch_;
    std::unique_ptr<message_types> netch_codes_;
    settings st_;

    bool stop_ = false;

    timers::queue_ptr timers_queue_;

    resource_clients resource_clients_;
    lease_callbacks_repository callbacks_;
    std::vector<peers_count_callback> peers_count_callbacks_;

    string node_id_;

    yplatform::util::string_uuid_generator uuids_generator_;
    std::atomic<bool> verbose_logging_;
    yplatform_log_wrapper yplatform_log_wrapper_;

    boost::asio::io_service* io_;
    boost::shared_ptr<boost::asio::io_service::strand> strand_;

    peers_count_accumulator peers_count_accumulator_;
};

}

#endif
