#include "nodes_registry.h"

namespace ylease {

nodes_registry::node::node(const string& address, boost::asio::io_service& io)
    : address(address), timer(io)
{
}

bool nodes_registry::node::expired() const
{
    return timer.expires_at() <= clock::now();
}

nodes_registry::nodes_registry(
    boost::asio::io_service& io,
    const update_callback& list_update_callback)
    : strand_(io), list_update_callback_(list_update_callback)
{
}

void nodes_registry::insert(
    const string& address,
    const node_id& id,
    const string& tag,
    const time_duration& inactive_timeout)
{
    strand_.post([this, self = shared_from_this(), address, id, tag, inactive_timeout]() {
        bool updated = false;
        auto& nodes = nodes_by_tag_[tag];
        auto it = nodes.find(id);
        if (it == nodes.end())
        {
            updated = true;
            it = nodes
                     .emplace(
                         std::piecewise_construct,
                         std::forward_as_tuple(id),
                         std::forward_as_tuple(address, strand_.get_io_service()))
                     .first;
        }
        auto& node = it->second;

        if (!updated && node.address != address)
        {
            node.address = address;
            updated = true;
        }

        node.timer.expires_from_now(inactive_timeout);
        node.timer.async_wait(strand_.wrap(std::bind(&nodes_registry::cleanup, self, id, tag)));
        if (updated)
        {
            yplatform::safe_call(list_update_callback_, tag, make_address_list(nodes));
        }
    });
}

void nodes_registry::cleanup(const node_id& id, const string& tag)
{
    auto nodes_it = nodes_by_tag_.find(tag);
    if (nodes_it == nodes_by_tag_.end()) return;
    auto& nodes = nodes_it->second;

    auto node_it = nodes.find(id);
    if (node_it == nodes.end()) return;
    auto& node = node_it->second;

    if (node.expired())
    {
        nodes.erase(node_it);
        yplatform::safe_call(list_update_callback_, tag, make_address_list(nodes));
    }
    if (nodes.empty())
    {
        nodes_by_tag_.erase(nodes_it);
    }
}

address_list nodes_registry::make_address_list(const id_to_node_map& nodes)
{
    address_list addresses;
    addresses.reserve(nodes.size());
    for (auto& pair : nodes)
    {
        addresses.push_back(pair.second.address);
    }
    return addresses;
}

}
