#include "resource_client.h"
#include "messages.h"

#include "time_millis.h"

///@todo think about
#define EXPIRE_TIME_DIVISOR 7

#define LOG_ENABLED log_func_
#define RES_STREAM_LOG()                                                                           \
    if (LOG_ENABLED) L_STREAM(log_func_) << "<" << (resource_name()) << "> "

using namespace ylease;

resource_client::resource_client(
    string node_id,
    const string& resource_name,
    const string& tag,
    const unsigned acquire_lease_timeout,
    const unsigned max_lease_time,
    const size_t arbiters_count,
    timers::queue_interface& timers_queue,
    const send_prepare_callback& send_prepare_func,
    const send_accept_callback& send_accept_func,
    const resource_win_callback& win_callback,
    const resource_lose_callback& lose_callback,
    const log_function_t& log_func)
    : timers_queue_(timers_queue)
    , send_prepare_func_(send_prepare_func)
    , send_accept_func_(send_accept_func)
    , win_callback_(win_callback)
    , lose_callback_(lose_callback)
    , log_func_(log_func)
    , read_only_(false)
    , vote_round_state_(arbiters_count / 2 + 1)
{
    if (node_id.empty())
    {
        throw std::domain_error("can't create node resource with empty node id");
    }
    if (resource_name.empty())
    {
        throw std::domain_error("can't create node resource with empty name");
    }

    settings_ = { node_id,
                  resource_name,
                  tag,
                  milliseconds(acquire_lease_timeout),
                  milliseconds(max_lease_time) };

    highest_known_ballot_ = -1;
    next_expire_time_ = 0;

    winner_ = false;
    win_ballot_ = -1;
    last_busy_ballot_ = -1;
}

resource_client::~resource_client()
{
}

void resource_client::fini()
{
    stop_acquire_lease();
}

bool resource_client::is_mine() const
{
    scoped_lock lock(mux_);
    return winner_;
}

bool resource_client::active() const
{
    scoped_lock lock(mux_);
    return active_impl();
}

bool resource_client::active_impl() const
{
    return !vote_round_state_.idle();
}

void resource_client::start_acquire_lease()
{
    RES_STREAM_LOG() << "start_acquire_lease";

    scoped_lock lock(mux_);
    read_only_ = false;

    start_proposing(lock);
}

void resource_client::start_read_only()
{
    RES_STREAM_LOG() << "start_read_only";

    scoped_lock lock(mux_);
    read_only_ = true;

    start_proposing(lock);
}

void resource_client::stop_acquire_lease()
{
    RES_STREAM_LOG() << "stop_acquire_lease";
    scoped_lock lock(mux_);

    vote_round_state_.set_state_idle();

    // drop leadership, if leader
    winner_ = false;
    win_ballot_ = -1;

    cancel_expiry_timer();
    cancel_acquire_timer();

    // important: under lock
    lose_callback_();
}

void resource_client::update_acquire_value(const value& value)
{
    RES_STREAM_LOG() << "update_acquire_value value=" << value;

    scoped_lock lock(mux_);
    current_acquire_value_ = value;

    if (winner_) start_proposing(lock);
}

void resource_client::update_lease_time(const time_duration& time)
{
    RES_STREAM_LOG() << "update_lease_time time=" << time_traits::to_string(time);

    scoped_lock lock(mux_);
    settings_.max_lease_time = time;

    if (winner_) start_proposing(lock);
}

void resource_client::start_proposing(scoped_lock& lock)
{
    if (!active_impl())
    {
        cancel_acquire_timer();
        // will unlock
        start_preparing(lock);
        return;
    }
}

void resource_client::log_promise_message(const promise_msg& response)
{
    if (!response.is_busy() || response.lease.node == node_id())
    {
        RES_STREAM_LOG() << "promise for " << response.ballot << " from " << response.arbiter_id;
    }
    else
    {
        RES_STREAM_LOG() << "promise for " << response.ballot << ": resource busy"
                         << " from " << response.arbiter_id;
    }
}

void resource_client::update_highest_ballot(ballot_t ballot)
{
    if (highest_known_ballot_ < ballot)
    {
        highest_known_ballot_ = ballot;
    }
}

bool resource_client::promise(const promise_msg& response)
{
    if (response.resource_name != resource_name())
    {
        return false;
    }

    scoped_lock lock(mux_);

    if (!vote_round_state_.preparing() || response.ballot != vote_round_state_.ballot)
    {
        RES_STREAM_LOG() << "not relevant promise for ballot=" << response.ballot;
        return false;
    }

    log_promise_message(response);

    // we have to update highest ballot
    // at first promise message for restart on timeout with incremented ballot_
    update_highest_ballot(response.ballot);

    if (!vote_round_state_.add_ack(response.arbiter_id))
    {
        RES_STREAM_LOG() << "ERROR promise duplicate from " << response.arbiter_id;
        return false;
    }

    if (response.is_busy())
    {
        vote_round_state_.pick_lease(response.lease, response.accepted_ballot);
    }
    else
    {
        if (last_busy_ballot_ < response.lease.ballot)
        {
            RES_STREAM_LOG() << " updating current_acquire_value_=" << current_acquire_value_
                             << " to=" << response.lease.value;
            last_busy_ballot_ = response.lease.ballot;
            current_acquire_value_ = response.lease.value;
        }
    }

    // see if we have enough positive replies to advance
    if (vote_round_state_.ack_majority())
    {
        // will unlock to execute user's callback functions
        start_accepting(lock);
        return true;
    }
    return true;
}

bool resource_client::reject(const reject_msg& response)
{
    if (response.resource_name != resource_name())
    {
        return false;
    }

    scoped_lock lock(mux_);

    if (vote_round_state_.idle() || response.ballot != vote_round_state_.ballot)
    {
        return false;
    }

    // anywhere, highest_promised >= current ballot)
    update_highest_ballot(response.highest_promised);

    RES_STREAM_LOG() << "reject " << (vote_round_state_.preparing() ? "prepare" : "accept")
                     << " majority "
                     << " for " << response.ballot << " (top is " << response.highest_promised
                     << ")";

    // will unlock
    start_preparing(lock);

    return true;
}

bool resource_client::win(const accepted_msg& response)
{
    if (response.resource_name != resource_name())
    {
        return false;
    }

    scoped_lock lock(mux_);

    if (!vote_round_state_.accepting() || response.ballot != vote_round_state_.ballot)
    {
        RES_STREAM_LOG() << "not relevant proposal for ballot=" << response.ballot;
        return false;
    }

    if (!vote_round_state_.add_ack(response.arbiter_id))
    {
        RES_STREAM_LOG() << "ERROR win duplicate from " << response.arbiter_id;
        return false;
    }

    RES_STREAM_LOG() << "numAccepted: " << vote_round_state_.num_received << " from "
                     << response.arbiter_id;

    // see if we have enough positive replies to advance
    if (vote_round_state_.ack_majority())
    {
        // a majority have accepted our proposal, we have consensus
        if (next_expire_time_ - get_time_millis() > MIN_EXPIRE_TIME /*msec*/)
        {

            // @todo remove?
            cancel_acquire_timer();
            cancel_expiry_timer();

            // @todo process errors while no timer
            finish_acquiring(lock);
        }
        else
        {
            reset_winner();
        }
    }

    return true;
}

void resource_client::finish_acquiring(scoped_lock& /*lock*/)
{
    assert(vote_round_state_.accepting());

    time_millis now = get_time_millis();

    vote_round_state_.lease.duration = (next_expire_time_ - now) / EXPIRE_TIME_DIVISOR;
    RES_STREAM_LOG() << "win!.. duration: " << vote_round_state_.lease.duration << "/"
                     << (next_expire_time_ - now);

    reset_acquire_timer(milliseconds(vote_round_state_.lease.duration), vote_round_state_.ballot);
    reset_expiry_timer(milliseconds(next_expire_time_ - now), vote_round_state_.ballot);

    // prepare args for callbacks
    win_ballot_ = vote_round_state_.ballot;
    last_busy_ballot_ = win_ballot_;
    winner_ = true;
    lease leaseCopy = vote_round_state_.lease;
    ballot_t ballotCopy = win_ballot_;

    vote_round_state_.set_state_idle();

    // important: under lock
    win_callback_(leaseCopy, ballotCopy);
}

void resource_client::start_preparing(scoped_lock& lock)
{
    vote_round_state_.reset();
    // XXX why do init lease here???
    // vote_round_state_.lease.node = node_id();
    // vote_round_state_.lease.value = current_acquire_value_;
    vote_round_state_.set_state_preparing(highest_known_ballot_ + 1);
    reset_acquire_timer(settings_.acquire_lease_timeout, vote_round_state_.ballot);
    prepare_msg msg(resource_name(), vote_round_state_.ballot);
    msg.node = node_id();
    msg.tag = tag();
    msg.vote_interval = duration_cast<milliseconds>(
                            winner_ ? settings_.max_lease_time : settings_.acquire_lease_timeout)
                            .count();
    ballot_t ballotCopy = vote_round_state_.ballot;

    // unlock to log and to call to the send functor
    lock.unlock();
    RES_STREAM_LOG() << "prepare started ballot=" << ballotCopy;
    send_prepare_func_(msg);
}

void resource_client::start_accepting(scoped_lock& lock)
{
    assert(vote_round_state_.preparing());

    // if BUSY
    if (vote_round_state_.lease.node != "")
    {
        assert(vote_round_state_.answered_acceptors_max_ballot != -1);
        // if THEIR NODE
        if (vote_round_state_.lease.node != node_id())
        {
            reset_winner();
            auto leaseCopy = vote_round_state_.lease;
            auto ballotCopy = vote_round_state_.answered_acceptors_max_ballot;
            vote_round_state_.set_state_idle();

            RES_STREAM_LOG() << "resource busy ballot=" << ballotCopy << " node=" << leaseCopy.node
                             << " duration=" << leaseCopy.duration;

            // notify about current lease node
            // important: under lock
            win_callback_(leaseCopy, ballotCopy);
            return; // no point in getting someone else a lease,
                    // wait for OnAcquireLeaseTimeout
        }
    }
    else
    {
        assert(vote_round_state_.answered_acceptors_max_ballot == -1);
        // WARN can cause lose + win callback calls if master is lagging

        // important: under lock
        lose_callback_();
    }

    if (winner_ || !read_only_)
    {
        vote_round_state_.set_state_accepting();

        vote_round_state_.lease.node = node_id();
        vote_round_state_.lease.value = current_acquire_value_;
        vote_round_state_.lease.duration =
            read_only_ ? 0 : duration_cast<milliseconds>(settings_.max_lease_time).count();
        next_expire_time_ = get_time_millis() + vote_round_state_.lease.duration;
        accept_msg msg(resource_name(), vote_round_state_.ballot, vote_round_state_.lease);

        // unlock to log and to call to the send functor
        lock.unlock();
        RES_STREAM_LOG() << "accept started ballot=" << msg.ballot << " node=" << msg.lease.node
                         << " duration=" << msg.lease.duration;
        send_accept_func_(msg);
    }
    else
    {
        vote_round_state_.set_state_idle();
    }
}

bool resource_client::acquire_lease_timeout(ballot_t ballot)
{
    scoped_lock lock(mux_);

    RES_STREAM_LOG() << "acquire_lease_timeout timer_ballot=" << ballot
                     << " actual_ballot=" << vote_round_state_.ballot;

    // ignore not our ballot
    if (vote_round_state_.ballot != ballot)
    {
        return false;
    }

    // 1) expire timeout resets winner_
    // 2) active_impl - round timeout, can't determine if the resource is busy
    bool notify = !winner_ && (active_impl());

    // important: under lock
    if (notify) lose_callback_();

    // start new acquire round
    // will unlock
    start_preparing(lock);

    return true;
}

bool resource_client::expiry_lease_timeout(ballot_t ballot)
{
    scoped_lock lock(mux_);

    // if not winner
    if (!winner_)
    {
        return false;
    }
    // or if win epoch changed
    if (win_ballot_ != ballot)
    {
        RES_STREAM_LOG() << "not win ballot " << ballot << " expired (win ballot is " << win_ballot_
                         << ")";
        return false;
    }

    RES_STREAM_LOG() << "expired, winBallot " << win_ballot_;

    reset_winner();

    // cancel acquire timer before new round starting
    cancel_acquire_timer();

    // important: under lock
    lose_callback_();

    // restart round
    // will unlock
    start_preparing(lock);

    return true;
}

void resource_client::reset_expiry_timer(const time_duration& interval, ballot_t ballot)
{
    if (!expiry_timer_)
    {
        expiry_timer_ = timers_queue_.create_timer();
    }
    // this will cancel current waits
    expiry_timer_->async_wait(
        interval, boost::bind(&resource_client::expiry_lease_timeout, shared_from_this(), ballot));
}

void resource_client::reset_acquire_timer(const time_duration& interval, ballot_t ballot)
{
    if (!acquire_timer_)
    {
        acquire_timer_ = timers_queue_.create_timer();
    }
    // this will cancel current waits
    acquire_timer_->async_wait(
        interval, boost::bind(&resource_client::acquire_lease_timeout, shared_from_this(), ballot));
}

void resource_client::cancel_expiry_timer()
{
    if (expiry_timer_)
    {
        expiry_timer_->cancel();
    }
}

void resource_client::cancel_acquire_timer()
{
    if (acquire_timer_)
    {
        acquire_timer_->cancel();
    }
}

void resource_client::reset_winner()
{
    winner_ = false;
    win_ballot_ = -1;
}
