/**
 * @author: nordsturm
 * © Yandex LLC.
 */

#ifndef YMOD_LEASE_RESOURCE_CLIENT_H
#define YMOD_LEASE_RESOURCE_CLIENT_H

#include <set>
#include <boost/enable_shared_from_this.hpp>
#include "types_priv.h"
#include "messages.h"
#include "time_millis.h"
#include "timers_queue.h"
#include "logger.h"

namespace ylease {

// Dependencies injection, allows isolated testing, factorizes overall complexity.
typedef boost::function<void(const prepare_msg&)> send_prepare_callback;
typedef boost::function<void(const accept_msg&)> send_accept_callback;
typedef boost::function<void(const lease&, const ballot_t)> resource_win_callback;
typedef boost::function<void(void)> resource_lose_callback;

/** Node resource
 * NOTICE
 * set-timer and win-lose callbacks now are called under lock
 * send callbacks are called out of lock scope
 *
 * State machine (per resource), states: free, prepare, accept, busy.
 *
 * timers:
 *   - extend timeout - lease prolongation needed (for leader only)
 *   - acquire timeout - retry get lease
 */
class resource_client : public boost::enable_shared_from_this<resource_client>
{
public:
    resource_client(
        string node_id,
        const string& resource_name,
        const string& tag,
        const unsigned acquire_lease_timeout,
        const unsigned max_lease_time,
        const size_t arbiters_count,
        timers::queue_interface& timers_queue,
        const send_prepare_callback& send_prepare_func,
        const send_accept_callback& send_accept_func,
        const resource_win_callback& win_callback,
        const resource_lose_callback& lose_callback,
        const log_function_t& log_func);
    ~resource_client();

    void fini();

    bool is_mine() const;

    bool active() const;

    // api for start/stop resource leasing
    void start_acquire_lease();
    void start_read_only();
    void stop_acquire_lease();
    void update_acquire_value(const value& value);
    void update_lease_time(const time_duration& value);

    // events handlers, return false if message not relevant
    bool promise(const promise_msg& message);
    bool reject(const reject_msg& message);
    bool win(const accepted_msg& message);

    // are being called from timers
    bool acquire_lease_timeout(ballot_t ballot);
    bool expiry_lease_timeout(ballot_t ballot);

    const string& node_id() const
    {
        return settings_.node_id;
    }

    const string& resource_name() const
    {
        return settings_.resource_name;
    }

    const string& tag() const
    {
        return settings_.tag;
    }

private:
    void start_proposing(scoped_lock& lock);

    void reset_expiry_timer(const time_duration& interval, ballot_t ballot);
    void reset_acquire_timer(const time_duration& interval, ballot_t ballot);

    void cancel_expiry_timer();
    void cancel_acquire_timer();

    // flow-end functions, unlock and call timer-set and send callbacks
    void start_preparing(scoped_lock& lock);
    void start_accepting(scoped_lock& lock);
    void finish_acquiring(scoped_lock& lock);

    void timeout_impl(ballot_t ballot);

    bool active_impl() const;

    void update_highest_ballot(ballot_t ballot);

    void log_promise_message(const promise_msg& response);

    ///@todo think about external majority value receiving
    unsigned majority() const
    {
        return vote_round_state_.majority;
    }

    void reset_winner();

    mutable mutex mux_;
    string current_acquire_value_;

    ballot_t highest_known_ballot_;
    time_millis next_expire_time_;

    timers::queue_interface& timers_queue_;
    timers::timer_ptr expiry_timer_;
    timers::timer_ptr acquire_timer_;

    send_prepare_callback send_prepare_func_;
    send_accept_callback send_accept_func_;
    resource_win_callback win_callback_;
    resource_lose_callback lose_callback_;

    bool winner_;
    ballot_t win_ballot_;
    ballot_t last_busy_ballot_;
    log_function_t log_func_;

    bool read_only_;

    struct settings
    {
        string node_id;
        string resource_name;
        string tag;

        time_duration acquire_lease_timeout;
        time_duration max_lease_time;
    } settings_;

    // keeping track of messages during prepare and accept phases
    struct vote_round_state
    {
        lease lease;

        ballot_t ballot;

        unsigned num_received;
        unsigned num_rejected;

        std::set<string> answered_acceptors;
        ballot_t answered_acceptors_max_ballot; // for selecting top answer with top ballot

        enum class vote_state
        {
            idle,
            preparing,
            accepting
        } state;

        const unsigned majority;

        vote_round_state(unsigned _majority)
            : ballot(-1)
            , num_received(0)
            , num_rejected(0)
            , answered_acceptors_max_ballot(-1)
            , state(vote_state::idle)
            , majority(_majority)
        {
        }

        bool ack_majority()
        {
            return num_received >= majority;
        }

        bool add_ack(const string& arbiter_id)
        {
            if (answered_acceptors.count(arbiter_id))
            {
                return false;
            }
            answered_acceptors.insert(arbiter_id);
            num_received++;
            return true;
        }

        void pick_lease(const ylease::lease& _lease, ballot_t _ballot)
        {
            if (answered_acceptors_max_ballot < _ballot)
            {
                answered_acceptors_max_ballot = _ballot;
                lease = _lease;
            }
        }

        void reset_answers()
        {
            num_received = 0;
            num_rejected = 0;
            answered_acceptors.clear();
            answered_acceptors_max_ballot = -1;
        }

        void reset()
        {
            reset_answers();
            ballot = -1;
            lease.reset();
            state = vote_state::idle;
        }

        void set_state_preparing(ballot_t _ballot)
        {
            reset_answers();
            ballot = _ballot;
            state = vote_state::preparing;
        }

        void set_state_accepting()
        {
            reset_answers();
            state = vote_state::accepting;
        }

        void set_state_idle()
        {
            reset_answers(); // do not reset ballot
            lease.reset();
            state = vote_state::idle;
        }

        bool idle() const
        {
            return state == vote_state::idle;
        }

        bool preparing() const
        {
            return state == vote_state::preparing;
        }

        bool accepting() const
        {
            return state == vote_state::accepting;
        }

    } vote_round_state_;
};

}

#endif
