#include "resource_server.h"
#include "messages.h"

#include "time_millis.h"

using namespace ylease;

#define LOG_ENABLED log_func_
#define RES_STREAM_LOG(res)                                                                        \
    if (LOG_ENABLED) L_STREAM(log_func_) << "<" << (res) << "> "

resource_server::resource_server(
    const string& arbiter_id,
    const string& resource_name,
    timers::queue_interface& timers_queue,
    const log_function_t& log_func)
    : arbiter_id_(arbiter_id)
    , resource_name_(resource_name)
    , timers_queue_(timers_queue)
    , log_func_(log_func)
{
    if (resource_name.empty())
    {
        throw std::domain_error("can't create arbiter resource with empty name");
    }
    highest_promised = -1;
    switch_to_free();
}

resource_server::~resource_server()
{
}

void resource_server::fini()
{
    scoped_lock lock(mux_);
    cancel_timer();
}

void resource_server::switch_to_free()
{
    // no lock
    accepted_ballot = -1;
    accepted_lease.node = "";
    accepted_lease.duration = 0;
    accepted_expire_time = 0;
}

void resource_server::switch_to_busy(ballot_t ballot, const lease& lease)
{
    // no lock
    accepted_ballot = ballot;
    accepted_lease = lease;
    accepted_lease.ballot = ballot;
    accepted_expire_time = get_time_millis() + lease.duration;
}

bool resource_server::accepted_expired(time_millis now)
{
    return accepted_expire_time < now;
}

time_millis resource_server::accepted_expires(time_millis now)
{
    return accepted_expire_time > now ? accepted_expire_time - now : 0;
}

bool resource_server::is_busy() const
{
    scoped_lock lock(mux_);
    return accepted_ballot != -1;
}

bool resource_server::is_busy_impl() const
{
    return accepted_ballot != -1;
}

bool resource_server::is_free() const
{
    // lock in is_busy
    return !is_busy();
}

void resource_server::update_highest_ballot(ballot_t ballot)
{
    if (highest_promised < ballot)
    {
        highest_promised = ballot;
    }
}

void resource_server::prepare(
    const prepare_msg& request,
    const promise_callback& promise_send_callback,
    const reject_callback& reject_send_callback)
{
    if (request.resource_name != resource_name_)
    {
        return;
    }
    scoped_lock lock(mux_);
    if (is_busy_impl() && accepted_expired(get_time_millis()))
    {
        cancel_timer();
        RES_STREAM_LOG(resource_name_)
            << "handlePrepare lease with ballot " << accepted_ballot << " expiried";
        timeout_impl(accepted_ballot);
    }

    if (request.ballot < highest_promised)
    {
        RES_STREAM_LOG(resource_name_)
            << "prepare rejected with ballot:" << request.ballot << " / " << highest_promised;
        reject_msg msg(arbiter_id_, resource_name_, request.ballot, highest_promised);
        lock.unlock();
        reject_send_callback(msg);
    }
    else
    {
        update_highest_ballot(request.ballot);
        lease lease_copy = accepted_lease;
        lease_copy.duration = accepted_expires(get_time_millis());
        promise_msg msg(arbiter_id_, resource_name_, request.ballot, accepted_ballot, lease_copy);
        lock.unlock();
        RES_STREAM_LOG(resource_name_)
            << "promise with ballot:" << request.ballot << " with accepted " << accepted_lease.node
            << " in " << lease_copy.duration;
        promise_send_callback(msg);
    }
}

void resource_server::accept(
    const accept_msg& request,
    const accepted_callback& accepted_cb,
    const reject_callback& reject_cb)
{
    if (request.resource_name != resource_name_)
    {
        return;
    }
    scoped_lock lock(mux_);
    if (is_busy_impl() && accepted_expired(get_time_millis()))
    {
        cancel_timer();
        RES_STREAM_LOG(resource_name_)
            << "handleAccept lease with ballot " << accepted_ballot << " expiried";
        timeout_impl(accepted_ballot);
    }

    if (request.ballot != highest_promised)
    {
        reject_msg msg(arbiter_id_, resource_name_, request.ballot, highest_promised);
        lock.unlock();
        reject_cb(msg);
    }
    else
    {
        update_highest_ballot(highest_promised + 1);

        switch_to_busy(request.ballot, lease(request.lease));

        reset_timer(milliseconds(request.lease.duration), accepted_ballot);
        accepted_msg msg(arbiter_id_, resource_name_, request.ballot, accepted_lease);
        lock.unlock();
        accepted_cb(msg);
    }
}

void resource_server::timeout(ballot_t ballot)
{
    scoped_lock lock(mux_);
    timeout_impl(ballot);
}

void resource_server::timeout_impl(ballot_t ballot)
{
    if (is_busy_impl() && ballot == accepted_ballot)
    {
        RES_STREAM_LOG(resource_name_)
            << "timeout lease with ballot " << accepted_ballot << " expiried";
        switch_to_free();
    }
}

void resource_server::reset_timer(const time_duration& interval, ballot_t ballot)
{
    if (!timer_)
    {
        timer_ = timers_queue_.create_timer();
    }
    // this will cancel previous waits
    timer_->async_wait(
        interval, boost::bind(&resource_server::timeout, shared_from_this(), ballot));
}

void resource_server::cancel_timer()
{
    if (timer_)
    {
        timer_->cancel();
    }
}
